/*
 * This software is governed by the CeCILL-B license under French law and
 * abiding by the rules of distribution of free software.  You can  use, 
 * modify and/ or redistribute the software under the terms of the CeCILL-B
 * license as circulated by CEA, CNRS and INRIA at the following URL
 * "http://www.cecill.info" or the LICENCE.txt file present in this project.
*/

#ifndef HRBF_ORIENTED_BBOX_HPP__
#define HRBF_ORIENTED_BBOX_HPP__

// If defined enable bbox constructions visualitions with opengl
// (white points are dichotomic steps, colored points are newton iterations)
//#define GL_DEBUG_BBOX

#ifdef GL_DEBUG_BBOX
    #include "glsave.hpp"
    #include "port_glew.h"
#endif

#include "bbox.hpp"
#include "hrbf_core.hpp"

// -----------------------------------------------------------------------------

/// Shortcut to evaluate gradient and potential of an 3D HRBF
template< class Rbf >
float fngf(const HRBF_fit<float, 3, Rbf>& hrbf, const Point_cu& p, Vec3_cu& g)
{
    typename HRBF_fit<float, 3, Rbf>::Vector grad(g.x, g.y, g.z);
    float pot =  hrbf.eval_n_grad( typename HRBF_fit<float, 3, Rbf>::Vector(p.x, p.y, p.z),
                                   grad);
    g = Vec3_cu(grad(0), grad(1), grad(2));
    return pot;
}

// -----------------------------------------------------------------------------

/// Shortcut to evaluate the potential of an 3D HRBF
template< class Rbf >
float f(const HRBF_fit<float, 3, Rbf>& hrbf, const Point_cu& p)
{
    return hrbf.eval( typename HRBF_fit<float, 3, Rbf>::Vector(p.x, p.y, p.z) );
}

// -----------------------------------------------------------------------------

template< class Rbf >
float dichotomic_search(const Ray_cu& r,
                        float t0, float t1,
                        float target_iso,
                        const HRBF_fit<float, 3, Rbf>& hrbf,
                        float eps = 0.01f)
{
    float t = t0;
    float f0 = f( hrbf, r(t0) );
    float f1 = f( hrbf, r(t1) );

    if(f0 > f1){
        t0 = t1;
        t1 = t;
    }

    Point_cu p;
    for(unsigned short i = 0 ; i < 15; ++i)
    {
        t = (t0 + t1) * 0.5f;
        p = r(t);

        #ifdef GL_DEBUG_BBOX
        glColor3f(1.f, 1.f, 1.f);
        glVertex3f(p.x, p.y, p.z);
        #endif

        f0 = f( hrbf, p );

        if(f0 > target_iso){
            t1 = t;
            if((f0-target_iso) < eps) break;
        } else {
            t0 = t;
            if((target_iso-f0) < eps) break;
        }
    }
    return t;
}

// -----------------------------------------------------------------------------

/// Cast a ray and return the farthest point matching 'target_iso'.
/// We use newton iterations
/// @param start : origin of the ray must be inside the primitive
/// @param hrbf : field function we seek the iso-surface 'target-iso'
/// @param target_iso : iso_surface we want to reach while pushing the point start
/// @param hrbf_scale : overall scale of the hrbf.
/// Will be used for error detections
/// @param custom_dir : if true we don't follow the gradient but use 'dir'
/// defined by the user
/// @return the farthest point matching 'target_iso' along the ray.
template< class Rbf >
Point_cu push_point(const Point_cu& start,
                    const HRBF_fit<float, 3, Rbf>& hrbf,
                    float target_iso,
                    float hrbf_scale,
                    bool custom_dir = false,
                    const Vec3_cu& dir = Vec3_cu())
{
    Vec3_cu  grad;
    Vec3_cu  n_dir = dir.normalized();
    Vec3_cu  step;
    Point_cu res  = start;
    Point_cu prev = res;

    #ifdef GL_DEBUG_BBOX
    Vec3_cu cl= (dir + Vec3_cu(0.5f, 0.5f, 0.5f)) * 0.5f;
    glBegin(GL_POINTS);
    #endif

    for(int i = 0; i < 20; i++)
    {
        const float pot      = fngf( hrbf, res,  grad );
        const float pot_diff = fabsf( pot - target_iso);
        const float norm     = grad.safe_normalize();
        #ifdef GL_DEBUG_BBOX
        glColor3f(cl.x, cl.y, cl.z);
        glVertex3f(res.x, res.y, res.z);
        #endif

        // Check the norm are large enough
        if( custom_dir && n_dir.norm() > 0.00001f)
            step = n_dir;
        else if( norm > 0.00001f )
            step = grad;
        else{
            // Sometime moving a bit the evaluted point will help falling back
            // in an area where the gradient is not null
            res += Vec3_cu(0.00001f * hrbf_scale, 0.f, 0.f);
            continue;
        }

        float scale = (pot_diff / norm) * 0.4f;

        // check our step is not large compared to the overall scale of the object
        if( scale >  hrbf_scale * 4.f * target_iso ) {
            // Move a bit the point see if we get somewhere better...
            res += Vec3_cu(0.00001f * hrbf_scale, 0.f, 0.f);
            continue;
        }

        if( pot > target_iso)
        {
            Ray_cu r( prev, step);
            float t = dichotomic_search(r, 0.f, (res-prev).norm(), target_iso, hrbf);
            res = r( t );
            break;
        }

        prev = res;
        res = res + step * scale;

        if( pot_diff <= 0.01f || scale < 0.00001f)
            break;

    }
    #ifdef GL_DEBUG_BBOX
    glEnd();

    if( (res-start).norm( ) > target_iso * 2.f)
    {
        glBegin(GL_LINES);
        glVertex3f(start.x, start.y, start.z);
        glVertex3f(res.x, res.y, res.z);
        glEnd();
    }
    #endif

    return res;
}

// -----------------------------------------------------------------------------

/// Cast several rays along a square grid and return the farthest point which
/// potential is zero.
/// @param obbox : oriented bounding box which we add casted points to
/// @param org : origin point of the grid (top left corner)
/// @param x : extrimity point of the grid in x direction (top right corner)
/// @param y : extrimity point of the grid in y direction (bottom left corner)
/// @param hrbf_scale : overall scale of the hrbf.
/// @warning org, x and y parameters must lie in the same plane. Also this
/// (org-x) dot (org-y) == 0 must be true at all time. Otherwise the behavior
/// is undefined.
template< class Rbf >
void push_face(OBBox_cu& obbox,
               const Point_cu& org,
               const Point_cu& x,
               const Point_cu& y,
               const HRBF_fit<float, 3, Rbf>& hrbf,
               float target_iso,
               float hrbf_scale)
{
    int res = 8/*GRID_RES*/;

    Vec3_cu axis_x = x-org;
    Vec3_cu axis_y = y-org;
    Vec3_cu udir = (axis_x.cross(axis_y)).normalized();

    float len_x = axis_x.normalize();
    float len_y = axis_y.normalize();

    float step_x = len_x / (float)res;
    float step_y = len_y / (float)res;

    Transfo tr = obbox._tr.fast_invert();
    for(int i = 0; i < res; i++)
    {
        for(int j = 0; j < res; j++)
        {
            Point_cu p = org +
                         axis_x * (step_x * (float)i + step_x * 0.5f) +
                         axis_y * (step_y * (float)j + step_y * 0.5f);

            Point_cu res = push_point(p, hrbf, target_iso, hrbf_scale, true, udir);

            obbox._bb.add_point( tr * res);
        }
    }
}

// -----------------------------------------------------------------------------

template< class Rbf >
void get_samp_list(const HRBF_fit<float, 3, Rbf>& hrbf,
                   std::vector<Point_cu>& vec)
{
    vec.resize( hrbf.nb_nodes() );
    for(int i = 0; i < hrbf.nb_nodes(); ++i)
    {
        typename HRBF_fit<float, 3, Rbf>::Vector pos = hrbf.get_node(i);
        vec[i] = Point_cu(pos(0), pos(1), pos(2));
    }
}

// -----------------------------------------------------------------------------

template< class Rbf >
OBBox_cu compute_obbox(const HRBF_fit<float, 3, Rbf>& hrbf,
                       float target_iso = 2.f,
                       Transfo orientation = Transfo::identity() )
{
    #ifdef GL_DEBUG_BBOX
    GLEnabledSave gl_light(GL_LIGHTING  , true, false);
    GLEnabledSave gl_tex2d(GL_TEXTURE_2D, true, false);
    GLEnabledSave gl_tex3d(GL_TEXTURE_3D, true, false);
    GLPointSizeSave pt_size(5.f);
    #endif

    std::vector<Point_cu> samp_list;
    get_samp_list(hrbf, samp_list);

    OBBox_cu obbox;
    obbox._tr = orientation;
    Transfo bbox_tr_inv = obbox._tr.fast_invert();

    // Seek target_iso along samples normals of the HRBF
    for(unsigned i = 0; i < samp_list.size(); i++)
        obbox._bb.add_point(bbox_tr_inv * samp_list[i]);

    float bbox_scale = obbox._bb.lengths().get_max();
    for(unsigned i = 0; i < samp_list.size(); i++){
        Point_cu pt = push_point(samp_list[i], hrbf, target_iso, bbox_scale);
        obbox._bb.add_point(bbox_tr_inv * pt);
    }

#if 1
    // Push obbox faces by discretizing them into regulars 2d grid.
    // Eache point of the grid is pushed to the target iso
    std::vector<Point_cu> corners;
    /**
        @code

            6 +----+ 7
             /|   /|
          2 +----+3|
            |4+--|-+5
            |/   |/
            +----+
           0      1
        // Vertex 0 is pmin and vertex 7 pmax
        @endcode
    */
    for (int i = 0; i < 2; ++i)
    {
        // Get obox in world coordinates
        obbox._bb.get_corners(corners);
        for(int i = 0; i < 8; ++i)
            corners[i] = obbox._tr * corners[i];


        Point_cu list[6][3] = {{corners[2], corners[3], corners[0]}, // FRONT
                               {corners[0], corners[1], corners[4]}, // BOTTOM
                               {corners[3], corners[7], corners[1]}, // RIGHT
                               {corners[6], corners[2], corners[4]}, // LEFT
                               {corners[7], corners[6], corners[5]}, // REAR
                               {corners[6], corners[7], corners[2]}, // TOP
                              };

        // Pushing according a grid from the box's faces
        for(int i = 0; i < 6; ++i) {
            push_face(obbox, list[i][0], list[i][1], list[i][2], hrbf, target_iso, bbox_scale);
        }
    }
#endif

    return obbox;
}

#endif // HRBF_ORIENTED_BBOX_HPP__
