/*
 * This software is governed by the CeCILL-B license under French law and
 * abiding by the rules of distribution of free software.  You can  use, 
 * modify and/ or redistribute the software under the terms of the CeCILL-B
 * license as circulated by CEA, CNRS and INRIA at the following URL
 * "http://www.cecill.info" or the LICENCE.txt file present in this project.
*/

#include "obj_hrbf.hpp"

// -----------------------------------------------------------------------------

#include "g_vbo_primitives.hpp"
#include "glsave.hpp"
#include "glpick_fbo.hpp"


// -----------------------------------------------------------------------------

Obj_HRBF::Obj_HRBF() :
    Obj(),
    Node_implicit_surface(),
    _color_sample( Color::blue() ),
    _color_sample_selected( Color::yellow() ),
    _normal_length(1.f),
    _line_width(2.f),
    _point_size(5.f),
    _edit_mode(false)
{
    _render_type = Obj::POLYGONIZE;

    _picker_edit = new Edit_pick( this );

    _bo_normals_lines   = new GlBuffer_obj( GL_ARRAY_BUFFER );
    _bo_position_points = new GlBuffer_obj( GL_ARRAY_BUFFER );
    _bo_colors          = new GlBuffer_obj( GL_ARRAY_BUFFER );

    _samples.push_back( Sample(Vec3_cu(-1.f,0.f,0.f), Vec3_cu(-1.f,0.f,0.f)) );
    _samples.push_back( Sample(Vec3_cu(0.f,1.f,0.f), Vec3_cu(0.f,1.f,0.f)) );
    _samples.push_back( Sample(Vec3_cu(0.f,0.f,1.f), Vec3_cu(0.f,0.f,1.f)) );
    _samples.push_back( Sample(Vec3_cu(1.f,0.f,0.f), Vec3_cu(1.f,0.f,0.f)) );

    Samp_iterator it = _samples.begin();
    for(; it != _samples.end(); ++it) {
        _samples_ref.push_back( new Sample_ref(it, this) );
        Samp_ref_iterator last = --_samples_ref.end();
        (*last)->_id._it_ref = last;
    }

    update_gl_buffers();
}

// -----------------------------------------------------------------------------

Obj_HRBF::~Obj_HRBF()
{
    delete _picker_edit;
    delete _bo_normals_lines;
    delete _bo_position_points;
    delete _bo_colors;

    Samp_ref_iterator it = _samples_ref.begin();
    for(; it != _samples_ref.end(); ++it) {
        delete (*it);
    }
}

// -----------------------------------------------------------------------------

void Obj_HRBF::draw()
{
    if( !_edit_mode ) return;
    GLEnabledSave save_point(GL_POINT_SMOOTH, true, true );
    GLEnabledSave save_light(GL_LIGHTING,     true, false);
    GLEnabledSave save_tex  (GL_TEXTURE_2D,   true, false);

    GLLineWidthSave save_line_width( _line_width );
    GLPointSizeSave save_point_size( _point_size );

    glAssert( glPushMatrix() );
    {
        begin_draw_samples();
        draw_positions();
        draw_normals();
        end_draw_samples();
    }
    glAssert( glPopMatrix() );
}

// -----------------------------------------------------------------------------

float Obj_HRBF::f(const Vec3_cu& pos) const
{
    return _hrbf_solver.eval( HRBF_x3::Vector(pos.x, pos.y, pos.z) );
}

// -----------------------------------------------------------------------------

Vec3_cu Obj_HRBF::gf(const Vec3_cu& pos) const
{
    HRBF_x3::Vector g = _hrbf_solver.grad( HRBF_x3::Vector(pos.x, pos.y, pos.z) );
    return Vec3_cu(g(0), g(1), g(2));
}

// -----------------------------------------------------------------------------

float Obj_HRBF::fngf(const Vec3_cu& pos, Vec3_cu& grad) const
{
    HRBF_x3::Vector p(pos.x, pos.y, pos.z);

    HRBF_x3::Vector g = _hrbf_solver.grad( p );
    grad = Vec3_cu(g(0), g(1), g(2));

    return _hrbf_solver.eval( p );
}

// -----------------------------------------------------------------------------

void Obj_HRBF::draw_normals() const
{
    int size = _samples.size();
    _bo_colors->bind();
    glAssert( glColorPointer(4, GL_FLOAT, 0, 0) );
    _bo_normals_lines->bind();
    glAssert( glVertexPointer(3, GL_FLOAT, 0, 0) );
    glAssert( glDrawArrays(GL_LINES, 0, size*2)  );
}

// -----------------------------------------------------------------------------

void Obj_HRBF::draw_positions() const
{
    int size = _samples.size();
    _bo_colors->bind();
    // Since color are also used for drawing line primitive we need
    // to stride. The color is present twice for a point in the buffer obj
    glAssert( glColorPointer(4, GL_FLOAT, sizeof(Color)*2, 0) );
    _bo_position_points->bind();
    glAssert( glVertexPointer(3, GL_FLOAT, 0, 0) );
    glAssert( glDrawArrays(GL_POINTS, 0, size) );
}

// -----------------------------------------------------------------------------

void Obj_HRBF::begin_draw_samples() const
{
    glAssert( glPointSize(10.f) );
    glAssert( glEnableClientState(GL_VERTEX_ARRAY) );
    glAssert( glEnableClientState(GL_COLOR_ARRAY) );
}

// -----------------------------------------------------------------------------

void Obj_HRBF::end_draw_samples() const
{
    _bo_colors->unbind();
    glAssert( glVertexPointer(3, GL_FLOAT, 0, 0) );
    glAssert( glColorPointer (4, GL_FLOAT, 0, 0) );

    glAssert( glDisableClientState(GL_VERTEX_ARRAY) );
    glAssert( glDisableClientState(GL_COLOR_ARRAY) );
}

// -----------------------------------------------------------------------------

void Obj_HRBF::select_sample(const Samp_id& id, bool s)
{
    Color* ptr = 0;
    _bo_colors->map_to(ptr, GL_WRITE_ONLY);
    Color cl = s ? _color_sample_selected : _color_sample;
    ptr[id._gl_buff_idx * 2 + 0] = cl;
    ptr[id._gl_buff_idx * 2 + 1] = cl;
    _bo_colors->unmap();
    ptr = 0;
}

// -----------------------------------------------------------------------------

void Obj_HRBF::set_sample(const Samp_id& id, const Sample& samp)
{
    (*id._it) = samp;

    int gl_idx = id._gl_buff_idx;

    Vec3_cu* pos_ptr    = 0;
    Vec3_cu* normal_ptr = 0;
    _bo_position_points->map_to( pos_ptr   , GL_WRITE_ONLY );
    _bo_normals_lines->  map_to( normal_ptr, GL_WRITE_ONLY );

    pos_ptr[gl_idx]= samp._position;

    normal_ptr[gl_idx * 2 + 0] = samp._position;
    normal_ptr[gl_idx * 2 + 1] = samp._position + samp._normal * _normal_length;

    _bo_normals_lines->unmap();
    _bo_position_points->unmap();
    update_hrbf_evaluator();
}

// -----------------------------------------------------------------------------

void Obj_HRBF::delete_sample(Samp_id id)
{
    // unselected
    select_sample( id, false );

    delete (*id._it_ref);
    _samples_ref.erase( id._it_ref );
    _samples.    erase( id._it     );

    update_gl_buffers();
}

// -----------------------------------------------------------------------------

Sample_ref* Obj_HRBF::add_sample(const Sample& samp)
{
    _samples.push_back( samp );

    Sample_ref* samp_ref_ptr = 0;
    samp_ref_ptr = new Sample_ref(--_samples.end(), this);
    _samples_ref.push_back( samp_ref_ptr );
    Samp_ref_iterator last = --_samples_ref.end();
    (*last)->_id._it_ref = last;

    update_gl_buffers();

    return samp_ref_ptr;
}

// -----------------------------------------------------------------------------

void Obj_HRBF::update_gl_buffers()
{
    unsigned size = _samples.size();
    std::vector<Vec3_cu> buff_nodes  (size    );
    std::vector<Vec3_cu> buff_normals(size * 2);
    std::vector<Color>   buff_colors (size * 2);

    Samp_iterator     it          = _samples.    begin();
    Samp_ref_iterator it_samp_ref = _samples_ref.begin();
    assert( _samples_ref.size() == _samples.size() );
    _map_gl_idx_to_samp_it.resize( _samples.size() );
    for(int i = 0; it != _samples.end(); ++it, ++it_samp_ref, ++i)
    {
        Vec3_cu p = it->_position;
        Vec3_cu n = it->_normal;

        buff_nodes  [i] = p;
        buff_normals[i*2 + 0] = p;
        buff_normals[i*2 + 1] = p + n * _normal_length;
        buff_colors [i*2 + 0] = _color_sample;
        buff_colors [i*2 + 1] = _color_sample;

        (*it_samp_ref)->_id._gl_buff_idx = i;
        _map_gl_idx_to_samp_it[i] = it_samp_ref;
    }

    _bo_position_points->set_data(size * 3, &(buff_nodes[0]), GL_STATIC_DRAW);
    _bo_normals_lines  ->set_data(size * 3 * 2/*two points for a line*/, &(buff_normals[0]), GL_STATIC_DRAW);
    _bo_colors         ->set_data(size * 4 * 2/*two points for a line*/, &(buff_colors [0]), GL_STATIC_DRAW);

    update_hrbf_evaluator();
}

// -----------------------------------------------------------------------------

void Obj_HRBF::update_hrbf_evaluator()
{
    int size = _samples.size();
    std::vector<HRBF_x3::Vector> points (size);
    std::vector<HRBF_x3::Vector> normals(size);

    Samp_iterator it = _samples.begin();
    for(int i = 0; it != _samples.end(); ++it, ++i) {
        Vec3_cu pos = it->_position;
        Vec3_cu nor = it->_normal;
        points [i] = HRBF_x3::Vector(pos.x, pos.y, pos.z);
        normals[i] = HRBF_x3::Vector(nor.x, nor.y, nor.z);
    }

    _hrbf_solver.hermite_fit(points, normals);
}

// =============================================================================
// Obj_hrbf internal's class for Picking
// =============================================================================

Obj_HRBF::Edit_pick::Edit_pick(Obj_HRBF* ptr) :
    Picker( new Obj_hrbf_io( ptr ) ),
    _obj_hrbf(ptr)
{

}

// -----------------------------------------------------------------------------

bool Obj_HRBF::Edit_pick::select(Data_set& res,
                                 const Vec2i_cu& pos,
                                 const Params& p)
{
    GlPick_FBO* pick = p._fbo_picker;

    Obj::Use_shader save_shaders(false);

    Transfo tr  = _obj_hrbf->global_frame();
    Transfo eye = p.cam.get_eye_transfo();

    glAssert( glPushMatrix() );
    glAssert( glLoadMatrixf( (eye * tr).transpose().m ) );

    Transfo mat = p.cam.get_proj_transfo() * eye * tr;
    pick->begin(pos.x, pos.y, mat.transpose().m );
    {
        GLEnabledSave save_color_mat(GL_COLOR_MATERIAL, true, false);
        GLEnabledSave save_point(GL_POINT_SMOOTH, true, false);
        GLEnabledSave save_light(GL_LIGHTING,     true, false);
        GLEnabledSave save_tex  (GL_TEXTURE_2D,   true, false);

        GLLineWidthSave save_line_width( _obj_hrbf->_line_width * 10.f );
        GLPointSizeSave save_point_size( _obj_hrbf->_point_size * 10.f );

        _obj_hrbf->begin_draw_samples();
        pick->set_name( 0 );
        _obj_hrbf->draw_positions();
        pick->set_name( 1 );
        _obj_hrbf->draw_normals();
        _obj_hrbf->end_draw_samples();
    }
    glAssert( glPopMatrix() );
    GlPick_FBO::Ids ids = pick->end();
    int id = ids._primitive_id;

    assert(id > -2);
    assert(id < (int)_obj_hrbf->_samples_ref.size());
    res.clear();
    if( id > -1) res.insert( *(_obj_hrbf->_map_gl_idx_to_samp_it[id]) );

    return !res.empty();
}

// -----------------------------------------------------------------------------

static void select_func(Data* d, bool s)
{
    if( d->type_data() == EData::HRBF_SAMPLE)
    {
        Sample_ref* samp = (Sample_ref*)d;
        // Ok this can potentially do a lot of map unmap so if too slow
        // map umap of the buffer object should be done at a higher level where
        // select_func is called actually.
        samp->_obj_hrbf->select_sample(samp->_id, s);
    }else{
        assert(false);
    }
}

void Obj_HRBF::Edit_pick::set_selected(const Data_set& set, bool s)
{
    Picker::set_selected_fun(set, s, select_func);
}

// -----------------------------------------------------------------------------

void Obj_HRBF::Edit_pick::set_selected_all(bool state)
{
    if(!state){
        if(_selection.empty()) return;
        _selection.clear();
    }

    Obj_HRBF::Samp_ref_iterator it = _obj_hrbf->_samples_ref.begin();
    for(; it != _obj_hrbf->_samples_ref.end(); ++it)
    {
        Sample_ref* samp = (*it);

        samp->set_selected( state );
        if(state) _selection.insert( samp );
    }
}

// -----------------------------------------------------------------------------

void Obj_HRBF::Edit_pick::transform(const Data_set& set, const Transfo& tr)
{
    Data_set::const_iterator it = set.begin();
    for(; it != set.end(); ++it)
    {
        Data* data = *it;
        if( data->type_data() == EData::HRBF_SAMPLE )
        {
            Sample_ref* samp = (Sample_ref*)data;
            Obj_HRBF::Samp_id id = samp->_id;

            Transfo t = _obj_hrbf->global_frame().fast_invert() * tr * _obj_hrbf->global_frame();
            Obj_HRBF::Sample new_samp( t * _obj_hrbf->get_sample_pos(id).to_point(),
                                       t * _obj_hrbf->get_sample_normal(id) );

            _obj_hrbf->set_sample(samp->_id, new_samp);
        }
        else
            assert(false);
    }
}

// -----------------------------------------------------------------------------

void Obj_HRBF::Edit_pick::save_transform(const Data_set& set,
                                         std::vector<Data_tr>& transfo)
{
    Data_set::const_iterator it_set = set.begin();
    transfo.clear();
    transfo.reserve( set.size() );
    for(; it_set != set.end(); ++it_set)
    {
        if( (*it_set)->type_data() == EData::HRBF_SAMPLE )
        {
            Sample_ref* samp = (Sample_ref*)(*it_set);
            transfo.push_back( std::make_pair(samp, samp->lcl_frame()) );
        }else
            assert(false);
    }

}

// -----------------------------------------------------------------------------

void Obj_HRBF::Edit_pick::load_transform(const std::vector<Data_tr>& transfo)
{
    for(unsigned i = 0; i < transfo.size(); ++i)
    {
        Data* d = transfo[i].first;
        if( (d)->type_data() == EData::HRBF_SAMPLE )
        {
            Sample_ref* samp = dynamic_cast<Sample_ref*>( d );
            samp->set( transfo[i].second );
        }else{
            assert(false);
        }
    }
}

// -----------------------------------------------------------------------------

static Obj_HRBF* g_remove_func_attr_obj_hrbf;
static void remove_func(Data* data)
{
    if( data->type_data() == EData::HRBF_SAMPLE)
    {
        Sample_ref* samp = dynamic_cast<Sample_ref*>(data);
        g_remove_func_attr_obj_hrbf->delete_sample( samp->_id );
        //delete samp;
    }
}

void Obj_HRBF::Edit_pick::remove(const Data_set& set)
{
    g_remove_func_attr_obj_hrbf = _obj_hrbf;
    Picker::remove_fun( set, remove_func);
    g_remove_func_attr_obj_hrbf = 0;
}

// -----------------------------------------------------------------------------

void Obj_HRBF::Edit_pick::hold()
{
    _obj_hrbf->_edit_mode = true;
}

// -----------------------------------------------------------------------------

void Obj_HRBF::Edit_pick::release()
{
    _obj_hrbf->_edit_mode = false;
}

// -----------------------------------------------------------------------------
