/*
 * This software is governed by the CeCILL-B license under French law and
 * abiding by the rules of distribution of free software.  You can  use, 
 * modify and/ or redistribute the software under the terms of the CeCILL-B
 * license as circulated by CEA, CNRS and INRIA at the following URL
 * "http://www.cecill.info" or the LICENCE.txt file present in this project.
*/

#ifndef OBJ_HRBF_HPP__
#define OBJ_HRBF_HPP__

#include "object.hpp"
#include "node_implicit_surface.hpp"
#include "obj_hrbf_io.hpp"

#include "color.hpp"
#include "hrbf_phi_funcs.hpp"
#include "hrbf_core.hpp"
#include "hrbf_oriented_bbox.hpp"
#include "bbox.hpp"

#include <deque>

class Sample_ref;

// -----------------------------------------------------------------------------

/**
 * @class Obj_HRBF
 * @brief An implicit surface build with hermite radial basis functions
 *
 * Features: adding/removing/moving samples plus drawing samples.
 *
 *
 */
class Obj_HRBF : public Obj, public Node_implicit_surface {
public:

    // =========================================================================
    /// @name Internal datas definitions
    // =========================================================================

    /// @brief a sample is a simple pair of position and normal
    struct Sample {

        Sample(const Vec3_cu& pos, const Vec3_cu& nor) : _position(pos), _normal(nor) {  }

    private:
        friend class Obj_HRBF; // <- Allow access only to the handler class
        Vec3_cu _position;
        Vec3_cu _normal;
    };

    typedef std::list<Sample_ref*> Container_samp_ref;
    typedef std::list<Sample     > Container_samp;

    typedef Container_samp::iterator     Samp_iterator;
    typedef Container_samp_ref::iterator Samp_ref_iterator;

    /// @brief Sample identifier is its iterator in the sample list and gl index
    struct Samp_id {

        Samp_id(Samp_iterator it_, int idx) : _it(it_), _gl_buff_idx(idx) { }

    private:
        friend class Obj_HRBF; // <- Allow access only to the handler class
        Samp_iterator _it;
        Samp_ref_iterator _it_ref;
        int _gl_buff_idx;
    };

    // =========================================================================

    Obj_HRBF();
    ~Obj_HRBF();

    EObj::Obj_t type_object() const { return EObj::IMPLICIT; }

    Data* clone() const { return 0; }

    // =========================================================================
    /// @name Evaluate scalar field
    // =========================================================================

    float f(const Vec3_cu& pos) const;
    Vec3_cu gf(const Vec3_cu& pos) const;
    float fngf(const Vec3_cu& pos, Vec3_cu& grad) const;

    BBox_cu get_bbox() const { return compute_obbox(_hrbf_solver, 0.5f)._bb; }

    // =========================================================================
    /// @name Drawing
    // =========================================================================

    /// Draw the HRBF samples
    void draw();

    /// Draw positions with a single draw call (avoid ambiguties with picking)
    /// @warning don't forget to enable gl client state for vertex and color
    void draw_normals() const;

    /// Draw positions with a single draw call (avoid ambiguties with picking)
    /// @warning don't forget to enable gl client state for vertex and color
    void draw_positions() const;

    /// Setup gl state to draw HRBF's samples
    void begin_draw_samples() const;

    /// Restore gl states after drawing the samples
    /// (ie. draw_positions(), draw_normals())
    void end_draw_samples() const;

    /// Select a hrf sample and change is color
    /// @param id : sample identifier
    /// @param s : wether the sample is to be selected or unselected
    void select_sample(const Samp_id& id, bool s);

    // =========================================================================
    /// @name Accessors
    // =========================================================================

    int nb_samples() const { return _samples.size(); }

    Vec3_cu get_sample_pos   (const Samp_id& id) const { return id._it->_position; }
    Vec3_cu get_sample_normal(const Samp_id& id) const { return id._it->_normal;   }

    void set_sample(const Samp_id& id, const Sample& samp);

    void delete_sample(Samp_id id);

    /// Add a sample with parameters 'samp'
    Sample_ref* add_sample(const Sample& samp);

    // =========================================================================
    /// @name GUI Interface
    // =========================================================================
    /**
     * @brief Picking the HRBF samples and editing their locations
     */
    class Edit_pick : public Picker {
    public:
        Edit_pick(Obj_HRBF *ptr);

        bool select(Data_set& res, const Vec2i_cu& pos, const Params& p);
        void set_selected(const Data_set& set, bool s);
        void set_selected_all(bool state);

        void transform(const Data_set& set, const Transfo& tr);
        void save_transform(const Data_set& set, std::vector<Data_tr>& transfo);
        void load_transform(const std::vector<Data_tr>& transfo);

        void remove(const Data_set& s);
        void set_parent(const Data_set& /*set*/, Data* /*parent*/){ }
        void paste(Data_set_copy& /*s*/){  }

        void hold();
        void release();

        //void set_active(Data* d){  }

        Obj_HRBF* _obj_hrbf;
    };

    /// Get the interface to manipulate the object
    Picker* set_edit_mode(){ return _picker_edit; }

private:
    // =========================================================================
    /// @name Tools
    // =========================================================================

    void update_gl_buffers();

    void update_hrbf_evaluator();

    // =========================================================================
    /// @name Attributes
    // =========================================================================

    Picker* _picker_edit;

    /// gl buffer to represent sample's normals with GL_LINES
    GlBuffer_obj* _bo_normals_lines;
    /// gl buffer to represent sample's positions with GL_POINTS
    GlBuffer_obj* _bo_position_points;
    /// gl buffer for sample's colors
    GlBuffer_obj* _bo_colors;

    std::vector<Samp_ref_iterator> _map_gl_idx_to_samp_it;

    /// List of samples (a pair of position and normal)
    Container_samp _samples;

    /// Samples references for the picking
    Container_samp_ref _samples_ref;

    Color _color_sample;
    Color _color_sample_selected;

    /// Length of the normal when drawn
    float _normal_length;
    float _line_width;
    float _point_size;

    bool _edit_mode;

    typedef HRBF_fit<float, 3, Rbf_pow3<float> /*Rbf_x_sqrt_x<float>*/ /*Rbf_thin_plate<float>*/ > HRBF_x3;

    /// Class in charge of computing the samples interpolation by
    /// solving a linear system of equations
    HRBF_x3 _hrbf_solver;
};

// =============================================================================

/**
 * @class Sample_ref
 * @brief A HRBF's sample reference for picking
 *
 */
class Sample_ref : public Data {
public:
    Sample_ref(Obj_HRBF::Samp_iterator it,
               Obj_HRBF* hrbf) :
        _id(it, -1),
        _obj_hrbf(hrbf)
    {  }

    Transfo frame() const
    {
        Vec3_cu pos = _obj_hrbf->get_sample_pos   ( _id );
        Vec3_cu nor = _obj_hrbf->get_sample_normal( _id );
        return _obj_hrbf->global_frame() * Transfo::coordinate_system(pos, nor);
    }

    Transfo lcl_frame() const
    {
        Vec3_cu pos = _obj_hrbf->get_sample_pos   ( _id );
        Vec3_cu nor = _obj_hrbf->get_sample_normal( _id );
        return Transfo::coordinate_system(pos, nor);
    }

    void set(const Transfo& tr) { _obj_hrbf->set_sample(_id, Obj_HRBF::Sample(tr.get_org(), tr.x()) ); }

    void set_selected(bool s) { _obj_hrbf->select_sample(_id, s); }

    EData::Data_t type_data() const { return EData::HRBF_SAMPLE; }

    Data* clone() const { return 0; }

    // -------------------------------------------------------------------------
    /// @name attributes
    // -------------------------------------------------------------------------

    Obj_HRBF::Samp_id _id;       ///< sample identifier in _obj_hrbf.
    Obj_HRBF*         _obj_hrbf; ///< The HRBF the sample belongs to
};

// =============================================================================

#endif // OBJ_HRBF_HPP__
