/*
 * This software is governed by the CeCILL-B license under French law and
 * abiding by the rules of distribution of free software.  You can  use, 
 * modify and/ or redistribute the software under the terms of the CeCILL-B
 * license as circulated by CEA, CNRS and INRIA at the following URL
 * "http://www.cecill.info" or the LICENCE.txt file present in this project.
*/

#include "obj_hrbf_io.hpp"

#include "obj_hrbf.hpp"
#include "IO_selection.hpp"
#include "idx2_cu.hpp"
#include "std_utils.hpp"

// -----------------------------------------------------------------------------

Obj_hrbf_io::Obj_hrbf_io(Obj_HRBF* obj_hrbf_ptr) :
    _key_n(false),
    _obj_hrbf(obj_hrbf_ptr)
{ }

// -----------------------------------------------------------------------------

bool Obj_hrbf_io::mouse_press(QMouseEvent* e)
{
    // Add a sample with position on the clicked object and the pixel's normal
    // We sample the depth buffer to compute the normal
    if( e->button() == Qt::LeftButton && _key_n )
    {
        const Camera* cam = _io_selection->cam();

        // Read a 9x9 block of depths
        std::vector<float> z(9);
        float mx = e->x() - 1;
        float my = cam->height() - e->y() - 1;
        glReadPixels( mx, my,
                       3, 3,
                      GL_DEPTH_COMPONENT, GL_FLOAT, &(z[0]));

        // Unproject positions of the block to get world coordinates
        float z_avg = 0.f;
        std::vector<Point_cu> pos(9);
        for (int i = 0; i < 9; ++i)
        {
            float x = i % 3;
            float y = i / 3;
            z_avg += z[i];
            pos[i] = cam->un_project( Point_cu( mx + x, my + y, z[i]) );
        }

        /* Reorganize to get points organized in a circle:
         * 0 1 2
         * 7 8 3
         * 6 5 4
        */
        Utils::swap(pos[5], pos[3]);
        Utils::swap(pos[4], pos[8]);
        Utils::swap(pos[3], pos[7]);

        BBox_cu bbox = _obj_hrbf->get_bbox();
        // Compute normals of each triangle of the patch and average
        Vec3_cu normal( 0.f );
        Vec3_cu center( pos[8] );
        for(int i = 0; i < 6; ++i)
        {
            if( bbox.inside( Point_cu(pos[i]) ) && bbox.inside( Point_cu(pos[i+1]) ) )
            {
                Vec3_cu e0 = pos[i+1] - pos[i];
                Vec3_cu e1 = center   - pos[i];

                normal += e0.cross( e1 );
            }
        }

        normal = ( normal.norm() > 0.00001f ) ? normal.normalized() : -cam->get_dir();

        // If center outside the bbox means we didn't click on the object
        // and must find an arbitrary position and normal for the sample
        if( !bbox.inside( center.to_point() ) )
        {
            center = (bbox.pmax + bbox.pmin) * 0.5f;
            normal = -cam->get_dir();
        }

        _obj_hrbf->add_sample( Obj_HRBF::Sample( center, normal ) );
        return true;
    }

    return false;
}

// -----------------------------------------------------------------------------

bool Obj_hrbf_io::key_press(QKeyEvent* e)
{
    if( e->key() == Qt::Key_N ) _key_n = true;
    return false;
}

// -----------------------------------------------------------------------------

bool Obj_hrbf_io::key_release(QKeyEvent* e)
{
    if( e->key() == Qt::Key_N ) _key_n = false;
    return false;
}

// -----------------------------------------------------------------------------
