/*
 * This software is governed by the CeCILL-B license under French law and
 * abiding by the rules of distribution of free software.  You can  use, 
 * modify and/ or redistribute the software under the terms of the CeCILL-B
 * license as circulated by CEA, CNRS and INRIA at the following URL
 * "http://www.cecill.info" or the LICENCE.txt file present in this project.
*/

#ifndef _GEN_MESH_HPP__
#define _GEN_MESH_HPP__

#ifndef M_PI
#define M_PI (3.14159265358979323846f)
#endif

/** @brief Gen_mesh generates various mesh such as cones cylinders or spheres

  This toolkit only generates triangle meshes and line primitives

*/

// =============================================================================
namespace Gen_mesh {
// =============================================================================

/// @brief Simple structure to handle data generated by gen_mesh functions
/// for triangle meshes
struct Tri_Mesh_data{
    Tri_Mesh_data(int nb_vert, int nb_tri)
    {
        this->nb_tri = nb_tri;
        this->nb_vert = nb_vert;

        vertex     = new float[nb_vert*3u];
        normals    = new float[nb_vert*3u];
        index      = new int  [nb_tri *3u];
    }
    ~Tri_Mesh_data(){
        delete[] vertex;
        delete[] normals;
        delete[] index;
    }

    float* vertex;
    float* normals;
    int nb_vert;

    int* index;
    int  nb_tri;
};

// -----------------------------------------------------------------------------

/// @brief Simple structure to handle data generated by gen_mesh functions
/// for triangle meshes
struct Quad_Mesh_data{
    Quad_Mesh_data(int nb_vert, int nb_quad){
        this->nb_quad = nb_quad; this->nb_vert = nb_vert;
        vertex  = new float[nb_vert*3u];
        normals = new float[nb_vert*3u];
        index   = new int [nb_quad*4u];
    }
    ~Quad_Mesh_data(){
        delete[] vertex;
        delete[] normals;
        delete[] index;
    }

    float* vertex;
    float* normals;
    int nb_vert;

    int* index;
    int  nb_quad;
};

// -----------------------------------------------------------------------------

struct Line_data{
    Line_data(int nb_vert, int nb_line){
        this->nb_line = nb_line; this->nb_vert = nb_vert;
        vertex  = new float[nb_vert*3u];
        normals = new float[nb_vert*3u];
        index   = new int [nb_line*2u];
    }
    ~Line_data(){
        delete[] vertex;
        delete[] normals;
        delete[] index;
    }

    float* vertex;
    float* normals;
    int nb_vert;

    int* index;
    int  nb_line;
};

// =============================================================================

/// Generates a cylinder
/// @param length    cylinder's length
/// @param radius    cylinder's radius
/// @param nb_slice  Number of body slice
/// @param res       Number of vertices for the head and tail circle
/// @param cap       Close cylinder at each ends with a central vertex at
///                  length+radius
/// @return newly allocated mesh to be destroy by caller
Tri_Mesh_data* cylinder(float length,
                        float radius,
                        int nb_slice,
                        int res,
                        bool cap);

/// Generate a sphere centered about the origin
/// @param radius the radius of the sphere
/// @param res
/// @return newly allocated mesh to be destroyed by caller
Tri_Mesh_data* sphere(float radius,
                      int res);

/// Generate a cone
/// @param radius the radius at the bottom of the cone
/// @param res    number of horizontal and vertical slices
/// @return newly allocated mesh to be destroyed by caller
Tri_Mesh_data* cone(float radius,
                    float height,
                    int res);
// TODO: better cone generation with duplicate vertices for better shades

/// Generate a unit cube whose rear bottom left corner is at the origin.
Quad_Mesh_data* cube();

/// Circle contained in the xy plane centered about the origin
/// @param radius The circle's radius
/// @param res Number of vertices the circle is divided into
/// @param diameter the circle's circumference in radian (the default is full
/// circle)
/// @return newly allocated mesh to be destroyed by caller
Line_data* circle(float radius, int res, float diameter = 2.f * M_PI);

/// A grid contained in the xy plane centered about the origin
/// @param width The grid's width
/// @param height The grid's height
/// @param res_x Number of segments the grid width is divided into
/// @param res_y Number of segments the grid height is divided into
/// @return newly allocated mesh to be destroyed by caller
Line_data* grid(float width, float height, int res_x, int res_y);

/// Generate the cage shape of a cylinder
/// @param length    cylinder's length
/// @param radius    cylinder's radius
/// @param caps_res  number of segments for the circles forming the ends of the
/// cage
/// @param body_res number of segments forming the cylinder
Line_data* cylinder_cage(float length, float radius, int caps_res, int body_res);


}// END GEN_MESH NAMESPACE =====================================================

#endif // _GEN_MESH_HPP__
