/*
 * This software is governed by the CeCILL-B license under French law and
 * abiding by the rules of distribution of free software.  You can  use, 
 * modify and/ or redistribute the software under the terms of the CeCILL-B
 * license as circulated by CEA, CNRS and INRIA at the following URL
 * "http://www.cecill.info" or the LICENCE.txt file present in this project.
*/

#ifndef IO_INTERFACE_SKIN2_HPP__
#define IO_INTERFACE_SKIN2_HPP__

#include <QMouseEvent>
#include "main_window.hpp"
#include "portable_includes/port_glew.h"
#include "glsave.hpp"
#include "OGL_widget.hpp"
#include "intersection.hpp"
#include "common/qt_utils.hpp"

#ifndef M_PI
#define M_PI (3.14159265358979323846f)
#endif

/**
 * @brief The IO_interface_skin2 class handles camera and factorize some io
 * related functions
 *
 * Camera features:
 * @li shortcut to aligned point of views (front/top/etc.) with keys 1/3/7 and
 * ctrl button for opposite views.
 * @li translations back and forth proportionnal to center of view (step_cam()).
 * @li straff with shift and middle button.
 * @li rotation with different modes (turntable/expe/trackball).
 * @li keep orthogonal view and perspective always aligned with
 * the same viewport size.
 * @li enabling/disabling camera through '_enable_cam' boolean.
 *
 * IO features:
 * @li special keys states with _is_xxx_pushed attributes to check if multiples
 * keys are down at any time (mouse/keyboard events)
 *
 */
class IO_interface : public QObject {
    Q_OBJECT
public:
    IO_interface(OGL_widget* gl_widget) :
        _is_enter_pushed(false),
        _is_return_pushed(false),
        _is_ctrl_pushed(false),
        _is_alt_pushed(false),
        _is_tab_pushed(false),
        _is_shift_pushed(false),
        _is_space_pushed(false),
        _is_right_pushed(false),
        _is_left_pushed(false),
        _is_mid_pushed(false),
        _enable_cam(true),
        _gl_widget(gl_widget),
        _main_win(gl_widget->get_main_window()),
        _cam(gl_widget->camera()),
        _speed_cam(1.2f),
        _rot_speed(0.015f),
        _sign_rot(-1.f)
    {  }

    // -------------------------------------------------------------------------

    virtual ~IO_interface(){}

    // -------------------------------------------------------------------------
    /// @name Handling events
    // -------------------------------------------------------------------------

    virtual void mousePressEvent(QMouseEvent* event)
    {
        update_modifiers_keys( event->modifiers() );
        const int x = event->x();
        const int y = event->y();

        _cam_old_x = x;
        _cam_old_y = _cam->height() - y;

        _is_right_pushed = (event->button() == Qt::RightButton);
        _is_left_pushed  = (event->button() == Qt::LeftButton);
        _is_mid_pushed   = (event->button() == Qt::MidButton);

        if( _is_mid_pushed)
        {
            if( _cam->get_y().dot( Vec3_cu::unit_z() ) > 0.0f )
                _sign_rot = -1.f;
            else
                _sign_rot = 1.f;
        }
    }

    // -------------------------------------------------------------------------

    virtual void mouseReleaseEvent(QMouseEvent* event)
    {
        update_modifiers_keys( event->modifiers() );
        _is_right_pushed = (event->button() == Qt::RightButton) ? false : _is_right_pushed;
        _is_left_pushed  = (event->button() == Qt::LeftButton)  ? false : _is_left_pushed;
        _is_mid_pushed   = (event->button() == Qt::MidButton)   ? false : _is_mid_pushed;
    }

    // -------------------------------------------------------------------------

    virtual void mouseMoveEvent(QMouseEvent* event)
    {

        update_modifiers_keys( event->modifiers() );
        const int x = event->x();
        const int y = event->y();

        if( _enable_cam )
        {
            if( _is_mid_pushed && _is_shift_pushed ){
                do_camera_straff(x, y);
            }
            else if(_is_mid_pushed)
            {
                int dx = x - _cam_old_x;
                int dy = _cam->height() - y - _cam_old_y;
                if(false)        do_camera_rot_exp(dx, dy);        // Experimental
                else if( true )  do_camera_rot_turntable(dx, dy);  // Turntable
                else if( false ) do_camera_rot_trackball();        // Trackball
            }
        }

        _cam_old_x = x;
        _cam_old_y = _cam->height() - y;
    }

    // -------------------------------------------------------------------------

    /// Camera movements back and forth
    virtual void wheelEvent(QWheelEvent* event)
    {
        update_modifiers_keys( event->modifiers() );
        float numDegrees = event->delta() / 8.;
        float numSteps = numDegrees / 15.;

        if(event->buttons() == Qt::NoButton && _enable_cam )
        {
            float sign  = numSteps > 0 ? 1.f : -1.f;
            float width = _cam->get_ortho_zoom();

            float new_width = std::max( width - sign * width * _speed_cam * 0.1f, 0.1f);
            if(_cam->is_ortho()){
                _cam->set_ortho_zoom(new_width);
            }else
                _cam->update_pos( Vec3_cu(0.f, 0.f,  cam_step() * sign) );
        }
    }

    // -------------------------------------------------------------------------

    virtual void keyPressEvent(QKeyEvent* event)
    {
        update_modifiers_keys( event->modifiers() );
        // Check special keys
        switch( event->key() ){
        case Qt::Key_Control: _is_ctrl_pushed  = true; break;
        case Qt::Key_Alt:     _is_alt_pushed   = true; break;
        case Qt::Key_Tab:     _is_tab_pushed   = true; break;
        case Qt::Key_Shift:   _is_shift_pushed = true; break;
        case Qt::Key_Space:   _is_space_pushed = true; break;
        case Qt::Key_Enter:   _is_enter_pushed = true; break;
        case Qt::Key_Return:  _is_return_pushed= true; break;
        case Qt::Key_Left : /* screenshot(); */        break;
        }

        bool view_changed = true;
        if( _enable_cam )
        {
            bool triger = true;
            if( event->key() ==  Qt::Key_Up)
                _cam->update_pos( Vec3_cu(0.f, 0.f,  2. * cam_step() ) );
            else if( event->key() ==  Qt::Key_Down )
                _cam->update_pos( Vec3_cu(0.f, 0.f, -2. * cam_step() ) );
            else
                triger = false;

            if( !_is_ctrl_pushed )
            {
                if( event->key() == Qt::Key_1) {
                    front_view(); push_msge("front");
                } else if ( event->key() == Qt::Key_3 ) {
                    right_view(); push_msge("right");
                } else if ( event->key() == Qt::Key_7 ) {
                    top_view(); push_msge("top");
                } else
                    view_changed = false;
            }
            else
            {
                if( event->key() == Qt::Key_1) {
                    rear_view(); push_msge("rear");
                } else if ( event->key() == Qt::Key_3 ) {
                    left_view(); push_msge("left");
                } else if ( event->key() == Qt::Key_7 ) {
                    bottom_view(); push_msge("bottom");
                }else
                    view_changed = false;
            }

            if( event->key() == Qt::Key_5)
            {
                if( !_cam->is_ortho() ) adjust_ortho_zoom();
                else                    adjust_perspective_pos();
                _cam->set_ortho( !_cam->is_ortho() );
            }

            view_changed = view_changed || triger;
        }

        // Ok this is a bit hacky but it enables main window to see
        // keypress events and handle them. Be aware that a conflict will
        // appear if the same shortcut is used here and inside MainWindow.
        event->ignore();
    }

    // -------------------------------------------------------------------------

    virtual void keyReleaseEvent(QKeyEvent* event)
    {
        update_modifiers_keys( event->modifiers() );
        // Check special keys
        switch( event->key() ){
        case Qt::Key_Control: _is_ctrl_pushed  = false; break;
        case Qt::Key_Alt:     _is_alt_pushed   = false; break;
        case Qt::Key_Tab:     _is_tab_pushed   = false; break;
        case Qt::Key_Shift:   _is_shift_pushed = false; break;
        case Qt::Key_Space:   _is_space_pushed = false; break;
        case Qt::Key_Enter:   _is_enter_pushed = false; break;
        case Qt::Key_Return:  _is_return_pushed= false; break;
        }
        // Ok this is a bit hacky but it enables main window to see
        // keypress events and handle them. Be aware that a conflict will
        // appear if the same shortcut is used here and inside MainWindow.
        event->ignore();
    }

    // -------------------------------------------------------------------------
    /// @name Shortcut and accessors
    // -------------------------------------------------------------------------

    /// Shortcut function to get the gizmo
    Gizmo* gizmo(){ return _gl_widget->gizmo(); }

    /// Shortcut function to get the rendering context
    Render_context* ctx(){ return _gl_widget->ctx(); }

    /// Shortcut to get the mouse position in the QGLWidget
    /// (even when mouse tracking is disable and even in keyboards events)
    Vec2i_cu mouse_pos() const {
        QPoint p = _gl_widget->mapFromGlobal( QCursor::pos() );
        return Vec2i_cu(p.x(), p.y());
    }

    bool is_special_key_pushed() {
        return _is_ctrl_pushed  | _is_alt_pushed   | _is_tab_pushed   |
                _is_shift_pushed | _is_space_pushed | _is_enter_pushed |
                _is_return_pushed;
    }

    void update_modifiers_keys( const Qt::KeyboardModifiers& f )
    {
        _is_alt_pushed   = f.testFlag( Qt::AltModifier     );
        _is_ctrl_pushed  = f.testFlag( Qt::ControlModifier );
        _is_shift_pushed = f.testFlag( Qt::ShiftModifier   );
    }

    /// Draw a message on the viewport
    void push_msge(const QString& str){
        _gl_widget->_msge_stack->push(str, true);
    }

private:
    // -------------------------------------------------------------------------
    /// @name Private tools: handling camera movements
    // -------------------------------------------------------------------------
    void do_camera_straff(int x, int y)
    {
        const Vec3_cu pivot = _gl_widget->cam_pivot_pos();

        int px = x;
        int py = (_cam->height() - y);
        float z        = _cam->project( pivot.to_point() ).z;
        Point_cu p     = _cam->un_project( Point_cu(        px,         py, z) );
        Point_cu old_p = _cam->un_project( Point_cu(_cam_old_x, _cam_old_y, z) );
        Vec3_cu v = old_p - p;

        _cam->set_pos( _cam->get_pos() + v );
        _gl_widget->set_cam_pivot_user(  pivot + v );
    }

    // -------------------------------------------------------------------------

    /// Turntable rotations turns always around the global 'z' axis
    void do_camera_rot_turntable( int dx, int dy )
    {
        const Vec3_cu pivot = _gl_widget->cam_pivot_pos();
        dx *= _sign_rot;
        _cam->transform( Transfo::rotate( pivot, Vec3_cu::unit_z(),  dx * _rot_speed / M_PI) );
        _cam->transform( Transfo::rotate( pivot, _cam->get_x()    , -dy * _rot_speed / M_PI) );
    }

    // -------------------------------------------------------------------------

    void do_camera_rot_trackball()
    {

    }

    // -------------------------------------------------------------------------

    void do_camera_rot_exp(int dx, int dy)
    {
        const Vec3_cu pivot = _gl_widget->cam_pivot_pos();
        //rotation around x axis (to manage the vertical angle of vision)
        const float pitch = dy * _rot_speed / M_PI;
        //rotation around the y axis (vertical)
        const float yaw = -dx * _rot_speed / M_PI;
        //no roll
        _cam->update_dir(yaw, pitch, 0.f);

        if( _gl_widget->cam_pivot_type() != EOGL_widget::FREE )
        {
            // rotate around the pivot
            _gl_widget->update_pivot();
            Vec3_cu tcam  = _cam->get_pos();
            float d = (pivot - tcam).norm();
            _cam->set_pos(pivot - _cam->get_dir() * d);
        }
    }

    // -------------------------------------------------------------------------

    void right_view(){
        _cam->set_dir_and_up( -Vec3_cu::unit_x(),
                              Vec3_cu::unit_z() );

        Vec3_cu p = _cam->get_pos();
        Vec3_cu v = _gl_widget->cam_pivot_pos();
        _cam->set_pos( Vec3_cu( v.x + (p-v).norm(), v.y, v.z) );
    }

    void left_view(){
        _cam->set_dir_and_up(Vec3_cu::unit_x(),
                             Vec3_cu::unit_z());

        Vec3_cu p = _cam->get_pos();
        Vec3_cu v = _gl_widget->cam_pivot_pos();
        _cam->set_pos( Vec3_cu(v.x - (p-v).norm(), v.y, v.z) );
    }

    void rear_view(){
        _cam->set_dir_and_up(-Vec3_cu::unit_y(),
                             Vec3_cu::unit_z());

        Vec3_cu p = _cam->get_pos();
        Vec3_cu v = _gl_widget->cam_pivot_pos();
        _cam->set_pos( Vec3_cu(v.x, v.y + (p-v).norm(), v.z) );
    }

    void front_view(){
        _cam->set_dir_and_up(Vec3_cu::unit_y(),
                             Vec3_cu::unit_z());

        Vec3_cu p = _cam->get_pos();
        Vec3_cu v = _gl_widget->cam_pivot_pos();
        _cam->set_pos( Vec3_cu(v.x, v.y - (p-v).norm(), v.z) );
    }

    void top_view(){
        _cam->set_dir_and_up(-Vec3_cu::unit_z(),
                             Vec3_cu::unit_y());

        Vec3_cu p = _cam->get_pos();
        Vec3_cu v = _gl_widget->cam_pivot_pos();
        _cam->set_pos( Vec3_cu(v.x, v.y, v.z + (p-v).norm()) );
    }

    void bottom_view(){
        _cam->set_dir_and_up(Vec3_cu::unit_z(),
                             Vec3_cu::unit_y());

        Vec3_cu p = _cam->get_pos();
        Vec3_cu v = _gl_widget->cam_pivot_pos();
        _cam->set_pos( Vec3_cu(v.x, v.y, v.z - (p-v).norm()) );
    }

    // -------------------------------------------------------------------------

    /// Change frustum ortho zoom as to be as large as the
    /// plane going through the pivot in perspective mode
    void adjust_ortho_zoom()
    {
        _cam_old_near = _cam->get_near();
        const Vec3_cu pivot = _gl_widget->cam_pivot_pos();
        float dist_pivot = std::max( (_cam->get_pos() - pivot).norm(), 0.1f);
        float zoom = tanf( _cam->fovy() * 0.5f ) * dist_pivot;
        zoom *= ((float)_cam->width() / (float)_cam->height());
        _cam->set_ortho_zoom( zoom * 2.f );
        _cam->set_near( -10.f);
    }

    // -------------------------------------------------------------------------

    /// Change camera distance as to keep the image with same width as in ortho
    /// for the pivot point
    void adjust_perspective_pos()
    {
        const Vec3_cu pivot = _gl_widget->cam_pivot_pos();
        float zoom = _cam->get_ortho_zoom() * 0.5f;
        float dist = zoom / tanf( _cam->fovx() * 0.5f );
        Vec3_cu new_pivot = pivot - _cam->get_dir() * dist;
        if( (_cam->get_pos() - new_pivot).norm() >= (cam_step()-0.0001f) )
            _cam->set_pos( new_pivot );
        _cam->set_near( _cam_old_near );
    }

    // -------------------------------------------------------------------------

    /// Step linearly proportionnal to the distance of the camera pivot
    float cam_step()
    {
        Vec3_cu dir = _gl_widget->cam_pivot_pos() - _cam->get_pos();
        float step  = dir.norm() * 0.1f;
        return step * _speed_cam < 0.001f ? 0.1f : step * _speed_cam;
    }

    // -------------------------------------------------------------------------
    /// @name Accessors
    // -------------------------------------------------------------------------
public:

    const Camera* cam() const { return _cam; }

    // -------------------------------------------------------------------------
    /// @name Protected attributes
    // -------------------------------------------------------------------------
protected:
    bool _is_enter_pushed;  ///< is enter key pushed
    bool _is_return_pushed; ///< is return key pushed
    bool _is_ctrl_pushed;   ///< is control key pushed
    bool _is_alt_pushed;    ///< is alt key pushed
    bool _is_tab_pushed;    ///< is tabulation key pushed
    bool _is_shift_pushed;  ///< is shift key pushed
    bool _is_space_pushed;  ///< is space key pushed

    bool _is_right_pushed;  ///< is mouse right button pushed
    bool _is_left_pushed;   ///< is mouse left button pushed
    bool _is_mid_pushed;    ///< is mouse middle

    bool _enable_cam;       ///< enable/disable camera movements

    OGL_widget*  _gl_widget;
    Main_window* _main_win;

    Camera* _cam;           ///< camera pointer of the associated gl_widget

    // -------------------------------------------------------------------------
    /// @name Private attributes
    // -------------------------------------------------------------------------
private:
    float _cam_old_x;
    float _cam_old_y;
    float _cam_old_near;

    float _speed_cam;       ///< speed of the camera movements
    float _rot_speed;       ///< rotation speed of the camera
    float _sign_rot;        ///< the rotation sign in turntable mode
};

#endif // IO_INTERFACE_SKIN2_HPP__
