/*
 * This software is governed by the CeCILL-B license under French law and
 * abiding by the rules of distribution of free software.  You can  use, 
 * modify and/ or redistribute the software under the terms of the CeCILL-B
 * license as circulated by CEA, CNRS and INRIA at the following URL
 * "http://www.cecill.info" or the LICENCE.txt file present in this project.
*/

#ifndef IO_MANIPULATION_HPP__
#define IO_MANIPULATION_HPP__

#include "IO_selection.hpp"
#include "std_utils.hpp"

/**
 * @name IO_Manipulation
 * @brief Generic object selection/manipulation class
 */
class IO_Manipulation : public IO_Selection {
    Q_OBJECT
public:
    // -------------------------------------------------------------------------
    /// @name Events handling
    // -------------------------------------------------------------------------

    IO_Manipulation(OGL_widget* gl_widget, Scene_renderer* renderer) :
        IO_Selection(gl_widget, renderer),
        _disable_obj_translation(false),
        _last_click(-1, -1),
        _last_cog( Vec3_cu::zero() ),
        _last_cog_eye( Vec3_cu::zero() ),
        _axis_mode( EIO_Selection::VIEW_PLANE ),
        _inverse_value(false)
    {

    }

    // -------------------------------------------------------------------------

    virtual ~IO_Manipulation(){ }

    // -------------------------------------------------------------------------

    void mousePressEvent(QMouseEvent* e)
    {
        IO_Selection::mousePressEvent(e);

        const int x = e->x(), y = e->y();
        const Vec2i_cu pos(x, y);
        _last_click = pos;

        if( _selection.size() > 0)
        {

            if(_is_left_pushed && _transfo_mode != EIO_Selection::NONE)
                disable_movements();

            if( _is_right_pushed && _transfo_mode != EIO_Selection::NONE)
            {
                restore_selection_transfos();
                disable_movements();
                // untill right click release:
                _disable_obj_translation = true;
            }

            if(_is_right_pushed) save_selection_transfos();
        }
    }

    // -------------------------------------------------------------------------

    void mouseReleaseEvent(QMouseEvent* e)
    {
        IO_Selection::mouseReleaseEvent(e);
        if( _transfo_mode == EIO_Selection::TRANSLATION && _selection.size() > 0)
            gizmo()->show( false );

        if(!_is_right_pushed && _disable_obj_translation)
            _disable_obj_translation = false;
    }



    // -------------------------------------------------------------------------

    void mouseMoveEvent(QMouseEvent* e)
    {
        IO_Selection::mouseMoveEvent(e);
        const int x = e->x(), y = e->y();
        const Vec2i_cu pos(x, y);

        if( _selection.size() > 0 && !_disable_obj_translation)
        {
            if( _transfo_mode != EIO_Selection::NONE)
            {
                do_transform( pos );
            }
            else if( _is_right_pushed && (pos - _last_click).norm() > 5.f )
            {
                // We move enough the mouse while
                // right clicking to grab the object
                enable_transfo_mode( EIO_Selection::TRANSLATION );
                init_translation_from( pos );
            }
        }
    }

    // -------------------------------------------------------------------------

    void wheelEvent(QWheelEvent* e){ IO_Selection::wheelEvent(e); }

    // -------------------------------------------------------------------------

    void keyPressEvent(QKeyEvent* e)
    {
        using namespace EIO_Selection;
        IO_Selection::keyPressEvent(e);
        char l = Qt_utils::event_to_char( e );

        if( _transfo_mode != EIO_Selection::NONE )
        {
            if( _is_return_pushed || _is_enter_pushed  )
                disable_movements();
            else if( e->key() == Qt::Key_Backspace ){
                reset_custom_value();
                restore_selection_transfos();
            } else if( l >= 'x' && l <= 'z' ){
                setup_axis_mode( l );
                parse_and_move_object( ' ' /* <- hack to re-apply previous user input*/);
                if(_axis_mode == VIEW_PLANE) restore_selection_transfos();
                update_transform( mouse_pos() );
            } else if( (l >= '0' && l <= '9') || l == '-' || l == '.' || l == '/' )
                parse_and_move_object( l );
        }

        if( _selection.size() > 0 && !is_special_key_pushed() )
        {
            _axis_dir = get_axis( _axis_mode );

            if( (l == 'g' || l == 'r' || l == 's') && _transfo_mode == NONE)
                save_selection_transfos(); // We were not transforming obj ? -> save position

            if( l == 'g' && _transfo_mode != TRANSLATION)
            {
                enable_transfo_mode( TRANSLATION );
                if( _transfo_mode != NONE ){
                    restore_selection_transfos();
                    reset_custom_value();
                }
                init_translation_from( mouse_pos() );
            }
            else if( l == 'r' && _transfo_mode != ROTATION)
            {
                enable_transfo_mode( ROTATION );
                if( _transfo_mode != NONE ){
                    restore_selection_transfos();
                    reset_custom_value();
                }

                init_rotation_from( mouse_pos() );
            }
            else  if( l == 's' && _transfo_mode != SCALE)
            {
                enable_transfo_mode( SCALE );
                if( _transfo_mode != NONE ){
                    restore_selection_transfos();
                    reset_custom_value();
                }

                init_scale_from( mouse_pos() );
            }
        }
    }

    // -------------------------------------------------------------------------

    void keyReleaseEvent(QKeyEvent* e) { IO_Selection::keyReleaseEvent(e); }

    // -------------------------------------------------------------------------

    /// Opengl drawing of the current manipulation (scale/rotation/translation)
    void draw_manipulator() const
    {
        _line.draw( *_cam );

        using namespace EIO_Selection;

        glLineStipple(5, 0xAAAA);
        GLEnabledSave save_line(GL_LINE_STIPPLE, true, true);
        GLEnabledSave textures(GL_TEXTURE_2D, true, false);
        GLEnabledSave lighting(GL_LIGHTING  , true, false);
        GLLineWidthSave line( 1.f );
        GLPointSizeSave point( 10.f );

        glAssert( glMatrixMode(GL_PROJECTION) );
        glAssert( glPushMatrix() );
        glAssert( glLoadIdentity() );
        glAssert( glOrtho(0.f, (GLfloat)_cam->width(), 0.f, (GLfloat)_cam->height(), -1.f, 1.f) );
        {
            glAssert( glMatrixMode(GL_MODELVIEW) );
            glAssert( glPushMatrix() );
            glAssert( glLoadIdentity() );

            Vec2i_cu p( _last_pix_rot );
            if(_transfo_mode == ROTATION || _transfo_mode == SCALE)
            {
                if( _transfo_mode == SCALE )
                    p = mouse_pos();

                glColor3f(0.f, 0.f, 0.f);
                glBegin(GL_LINES);
                glVertex3f(_last_cog_eye.x, _last_cog_eye.y, 0.f);
                glVertex3f(p.x, _cam->height() - p.y, 0.f);
                glEnd();

                glColor3f(0.1f, 0.1f, 0.1f);
                glBegin(GL_POINTS);
                glVertex3f(_last_cog_eye.x, _last_cog_eye.y, 0.f);
                glVertex3f(p.x, _cam->height() - p.y, 0.f);
                glEnd();
            }
            glAssert( glPopMatrix() );
        }
        glAssert( glMatrixMode(GL_PROJECTION) );
        glAssert( glPopMatrix() );
        glAssert( glMatrixMode(GL_MODELVIEW) );
    }

private:
    // -------------------------------------------------------------------------
    /// @name Inner class
    // -------------------------------------------------------------------------

    /// Utility to draw an infinite line
    struct Axis_draw {

        Axis_draw() : _draw(false) { }

        void draw( const Camera& c ) const
        {
            if( !_draw ) return;

            GLEnabledSave tex(GL_TEXTURE_2D, true, false);
            GLEnabledSave lighting(GL_LIGHTING  , true, false);
            GLLineWidthSave line( 2.f );
            float d = (float)(c.get_far()- c.get_near()) * 2.f;
            Vec3_cu p0 = _pos + _dir *  d;
            Vec3_cu p1 = _pos + _dir * -d;
            _cl.set_gl_state();
            glBegin(GL_LINES);
            glVertex3f(p0.x, p0.y, p0.z);
            glVertex3f(p1.x, p1.y, p1.z);
            glEnd();
        }

        bool _draw;
        Vec3_cu _dir;
        Vec3_cu _pos;
        Color _cl;
    };

    // -------------------------------------------------------------------------
    /// @name Transformations
    // -------------------------------------------------------------------------

    /// @param pos : in qt coords
    void do_axis_proj_translation( const Vec2i_cu& pos )
    {
        Ray_cu  ray = _cam->cast_ray(pos.x, pos.y);
        Vec3_cu up  = _cam->get_y();
        Vec3_cu dir = _cam->get_dir();

        // Find a plane passing through axis_dir and as parrallel as possible to
        // the camera image plane
        Vec3_cu ortho = dir.cross(_axis_dir);
        Vec3_cu p_normal;
        if(ortho.norm() < 0.00001) p_normal = _axis_dir.cross( up  );
        else                       p_normal = _axis_dir.cross(ortho);

        const Vec3_cu org = _saved_cog;
        Inter::Plane axis_plane(p_normal, org);
        Inter::Line l_ray(ray._dir, ray._pos);
        Vec3_cu slide_point, dst;
        if( Inter::plane_line(axis_plane, l_ray, slide_point) )
            // Constraint onto on axis:
            // project on _axis_dir the intersection point dst
            dst = org + _axis_dir * _axis_dir.dot(slide_point-org);
        else
            dst = Vec3_cu::zero();

        Transfo tr = Transfo::translate( dst - transfo_center());
        _current_pick->transform( _selection, tr );
    }

    // -------------------------------------------------------------------------

    /// @param pos : in qt coords
    void do_view_plane_translation( const Vec2i_cu& pos )
    {
        Vec3_cu dst = _cam->un_project( Point_cu(pos.x,
                                                 _cam->height() - pos.y,
                                                 _last_cog_eye.z)
                                        );

        Transfo tr = Transfo::translate( dst - _last_cog );
        _current_pick->transform( _selection, tr );
        // Recompute _last_cog_xx
        _last_cog = transfo_center();
        _last_cog_eye = _cam->project( _last_cog.to_point() ).to_vector();
    }

    // -------------------------------------------------------------------------

    /// @param pos : in qt coords
    void do_translation( const Vec2i_cu& pos )
    {
        Vec2i_cu p = pos - _diff_pixel;

        if( _axis_mode != EIO_Selection::VIEW_PLANE)
            do_axis_proj_translation( p );
        else
            do_view_plane_translation( p );
    }

    // -------------------------------------------------------------------------

    void do_rotation( const Vec2i_cu& pos )
    {
        Vec2_cu qt_cog_eye = Vec2_cu( _last_cog_eye.x, _cam->height() - _last_cog_eye.y);
        Vec2_cu v0 = _last_pix_rot  - qt_cog_eye;
        Vec2_cu v1 = Vec2_cu( pos ) - qt_cog_eye;

        v0.normalize();
        v1.normalize();
        float z = Vec3_cu( v0, 0.f ).cross( Vec3_cu(v1, 0.f) ).z;
        float a = std::acos( v0.dot( v1 ) ) * ( z > 0.f ? 1.f : -1.f);

        float sign = _axis_dir.dot( _cam->get_dir() ) < 0.f  ? -1.f : 1.f;

        if(std::abs(a) > 0.0001f && v0.norm() > 0.0001f && v1.norm() > 0.0001f )
            _current_pick->transform( _selection, Transfo::rotate(transfo_center(), _axis_dir * sign, a)  );

        _last_pix_rot = Vec2_cu( pos );
    }

    // -------------------------------------------------------------------------

    void do_scale( const Vec2i_cu& pos )
    {
        Vec2_cu qt_cog_eye = Vec2_cu( _last_cog_eye.x, _cam->height() - _last_cog_eye.y);
        Vec2_cu v0 = _last_pix_rot  - qt_cog_eye;
        Vec2_cu v1 = Vec2_cu( pos ) - qt_cog_eye;
        float diff = v0.norm() - v1.norm();

        float val = 1.f - diff / _scale_len;

        restore_selection_transfos();
        _current_pick->transform( _selection, compute_scale( val )  );
    }

    // -------------------------------------------------------------------------

    /// Given mouse 'pos' apply the transformations according to the current
    /// activated mode (see '_transfo_mode' and '_axis_mode')
    /// @param pos : in qt coords
    void do_transform( const Vec2i_cu& pos )
    {
        switch( _transfo_mode){
        case EIO_Selection::TRANSLATION: do_translation( pos ); break;
        case EIO_Selection::ROTATION:    do_rotation( pos );    break;
        case EIO_Selection::SCALE:       do_scale( pos );       break;
        default: break;
        }
    }

    // -------------------------------------------------------------------------

    /// @param pos : in qt coords
    void update_transform( const Vec2i_cu& pos )
    {
        if( _transfo_mode == EIO_Selection::TRANSLATION)
            _last_cog = transfo_center();

        do_transform( pos );
    }

    // -------------------------------------------------------------------------

    /// Given the '_axis_mode' and '_transfo_mode' compute a scale of 'val'
    Transfo compute_scale( float val )
    {
        using namespace EIO_Selection;
        Transfo tr;

        Vec3_cu sc;
        switch( _axis_mode ){
        case GX: case LX: sc = Vec3_cu(val, 1.f, 1.f); break;
        case GY: case LY: sc = Vec3_cu(1.f, val, 1.f); break;
        case GZ: case LZ: sc = Vec3_cu(1.f, 1.f, val); break;
        default: break;
        }

        if( is_global( _axis_mode ) )
        {

            Vec3_cu t = transfo_center();
            tr = Transfo::scale( t, sc );
        }
        else if( is_local(_axis_mode) )
        {
            Transfo lcl = transfo_frame_from_selection();
            tr = lcl * Transfo::scale( sc ) * lcl.fast_invert();
        }
        else
        {
            Vec3_cu t = transfo_center();
            tr = Transfo::scale( t, val );
        }

        return tr;
    }

    // -------------------------------------------------------------------------

    /// Given the '_axis_mode' and '_transfo_mode' compute a transformation
    /// given the user value 'val'
    Transfo compute_transfo( float val, bool& do_transform)
    {
        using namespace EIO_Selection;
        Transfo tr = Transfo::identity();

        do_transform = true;
        if( _axis_mode != VIEW_PLANE )
        {
            if( _transfo_mode == TRANSLATION )
                tr = Transfo::translate( _axis_dir * val );
            else if( _transfo_mode == ROTATION )
                tr = Transfo::rotate(_saved_cog, _axis_dir, val * M_PI / 180.f);
            else if( _transfo_mode == SCALE )
                tr = compute_scale( val );
            else
                do_transform = false;
        }
        else
        {
            if( _transfo_mode == SCALE )
                tr = compute_scale( val );
            else
                do_transform = false;
        }

        return tr;
    }

    // -------------------------------------------------------------------------

    Vec3_cu get_axis( EIO_Selection::Axis_t a)
    {
        Vec3_cu axis = Vec3_cu::zero();
        Transfo tr = transfo_frame_from_selection();
        switch(a){
        case EIO_Selection::LX: axis = tr.x();            break;
        case EIO_Selection::GX: axis = Vec3_cu::unit_x(); break;
        case EIO_Selection::LY: axis = tr.y();            break;
        case EIO_Selection::GY: axis = Vec3_cu::unit_y(); break;
        case EIO_Selection::LZ: axis = tr.z();            break;
        case EIO_Selection::GZ: axis = Vec3_cu::unit_z(); break;
        default: axis = _cam->get_dir(); break;
        }

        return axis.normalized();
    }

    // -------------------------------------------------------------------------
    /// @name Parsing custom vals
    // -------------------------------------------------------------------------

    /// Convert a list of input character to a real number
    float compute_val( const std::list<char>& tokens )
    {
        float result = 0.f;
        if( tokens.size() == 0 )  return result;

        const std::list<char>& tmp_toks = tokens;
        std::list<char>::const_iterator it = tmp_toks.begin();

        std::string str_val;
        for(; it != tmp_toks.end(); ++it)
            str_val.append( &(*it), 1 );

        result = Std_utils::to_real<float>( str_val );
        return result;
    }

    // -------------------------------------------------------------------------

    /// Add letter to the list of parsed character and moving objects according
    /// to the value parsed so far
    void parse_and_move_object( char letter )
    {
        if(letter == '-'){
            if( _parsed_value.size() > 0 && *(_parsed_value.begin()) == '-' )
                _parsed_value.erase( _parsed_value.begin() );
            else
                _parsed_value.push_front( '-' );
        }
        else if( letter == '/')
            _inverse_value = !_inverse_value;

        if( _parsed_value.size() < 20)
        {
            if( letter >= '0' && letter <= '9')
                _parsed_value.push_back( letter);
            else if( letter == '.' && !Std_utils::exists(_parsed_value, '.') )
                _parsed_value.push_back( letter);
        }

        float val = compute_val( _parsed_value );
        if( _parsed_value.size() == 0 && _transfo_mode == EIO_Selection::SCALE)
            val = 1.f;

        if( _inverse_value ) val = 1.f / val;

        print_parsed_value( val );

        // Move obj
        bool do_transform;
        Transfo tr = compute_transfo( val, do_transform );
        if( do_transform )
        {
            restore_selection_transfos();
            _current_pick->transform( _selection, tr );
        }else
            reset_custom_value();
    }

    // -------------------------------------------------------------------------

    /// Print in the viewport custom value and type of transformation applied
    void print_parsed_value(float val, bool undef = false)
    {
        using namespace EIO_Selection;
        _gl_widget->_msge_stack->clear();
        QString msge;

        switch(_transfo_mode){
        case TRANSLATION: msge.append( "Translation | " ); break;
        case ROTATION:    msge.append( "Rotation| "     ); break;
        case SCALE:       msge.append( "Scale | "       ); break;
        default: break;
        }

        switch(_axis_mode){
        case LX: case LY: case LZ:  msge.append( "local " ); break;
        default: break;
        }

        switch(_axis_mode){
        case LX: case GX:  msge.append( "x: " ); break;
        case LY: case GY:  msge.append( "y: " ); break;
        case LZ: case GZ:  msge.append( "z: " ); break;
        case VIEW_PLANE:
            if(_transfo_mode == SCALE)
                msge.append( "x/y/z: " );
            else
                msge.append( "local view x/y: " );
            break;
        }

        if( undef ) msge.append( "undef" );
        else        msge.append( QString::number(val) );

        _gl_widget->_msge_stack->push( msge );
    }

    // -------------------------------------------------------------------------

    /// given an axis to swith to ('letter' == x, y or z)  activate the correct
    /// axis mode.
    void setup_axis_mode(char letter)
    {
        using namespace EIO_Selection;
        if( letter == 'x')
            _axis_mode = switch_axis_mode( _axis_mode, (int)GX);
        else if( letter == 'y')
            _axis_mode = switch_axis_mode( _axis_mode, (int)GY);
        else if( letter == 'z')
            _axis_mode = switch_axis_mode( _axis_mode, (int)GZ);
        else
            assert(false); // called with wrong letter

        _axis_dir   = get_axis( _axis_mode );
        // Set line drawing settings in the viewport
        _line._draw = _axis_mode != VIEW_PLANE;
        _line._dir  = _axis_dir;
        _line._pos  = _saved_cog;
        _line._cl   = get_axis_color( _axis_mode );
    }

    // -------------------------------------------------------------------------

    /// Given an axis mode 'in' give the next axis mode associated
    /// to 'mode' == GX, GY or GZ
    EIO_Selection::Axis_t switch_axis_mode( EIO_Selection::Axis_t in, int mode)
    {
        using namespace EIO_Selection;
        assert( mode == (int)GX || mode == (int)GY || mode == (int)GZ );
        // note: local axis always next to global in enumerant
        int out = -1;
        if( in == mode )
            out = (mode+1);
        else if( in == (mode+1))
            out = (int)EIO_Selection::VIEW_PLANE;
        else
            out = mode;

        return (EIO_Selection::Axis_t)out;
    }

    // -------------------------------------------------------------------------
    /// @name Tools
    // -------------------------------------------------------------------------

    void enable_transfo_mode( EIO_Selection::Transfo_t mode )
    {
        _transfo_mode = mode;
        _gl_widget->setMouseTracking(true);
        _enable_cam = false;
        gizmo()->show( false );
    }

    // -------------------------------------------------------------------------

    void reset_custom_value()
    {
        _parsed_value.clear();
        _inverse_value = false;
    }

    // -------------------------------------------------------------------------

    void disable_movements()
    {
        reset_custom_value();
        _transfo_mode = EIO_Selection::NONE;
        _axis_mode = EIO_Selection::VIEW_PLANE;
        _line._draw = false;
        _gl_widget->setMouseTracking(false);
        _enable_cam = true;
        init_gizmo();
    }

    // -------------------------------------------------------------------------

    void save_selection_transfos()
    {
        _saved_transfo_selection.clear();
        _current_pick->save_transform( _selection, _saved_transfo_selection);
        _saved_cog = transfo_center()/*cog( _selection )*/;
    }

    // -------------------------------------------------------------------------

    void restore_selection_transfos()
    {
        _current_pick->load_transform( _saved_transfo_selection );
    }

    // -------------------------------------------------------------------------

    /// initialize translation with mouse coordinates
    /// @param pos : in qt coords (0,0) at upper left corner
    void init_translation_from( const Vec2i_cu& pos )
    {
        _last_cog = transfo_center();
        _last_cog_eye = _cam->project( _last_cog.to_point() ).to_vector();

        Vec2i_cu qt_cog_eye( _last_cog_eye );
        qt_cog_eye.y = _cam->height() - qt_cog_eye.y;
        _diff_pixel = pos - qt_cog_eye;
    }

    // -------------------------------------------------------------------------

    /// initialize rotation with mouse coordinates
    /// @param pos : in qt coords (0,0) at upper left corner
    void init_rotation_from( const Vec2i_cu& pos )
    {
        _last_pix_rot = Vec2_cu( pos );

        // might want to use another variables than the var from translation
        _last_cog = transfo_center();
        _last_cog_eye = _cam->project( _last_cog.to_point() ).to_vector();
    }

    // -------------------------------------------------------------------------

    /// initialize scale with mouse coordinates
    /// @param pos : in qt coords (0,0) at upper left corner
    void init_scale_from( const Vec2i_cu& pos )
    {
        _last_pix_rot = Vec2_cu( pos );

        _last_cog = transfo_center();
        _last_cog_eye = _cam->project( _last_cog.to_point() ).to_vector();
        Vec2_cu qt_cog_eye = Vec2_cu( _last_cog_eye.x, _cam->height() - _last_cog_eye.y);
        _scale_len = (_last_pix_rot  - qt_cog_eye).norm();
    }

private:
    // -------------------------------------------------------------------------
    /// @name Attributes
    // -------------------------------------------------------------------------

    bool _disable_obj_translation;

    float _scale_len;

    /// Last mouse pixel for rotation transformations
    Vec2_cu _last_pix_rot;

    /// Last mouse position on press event
    Vec2i_cu _last_click;

    /// Save difference between user click and cog projected to the screen.
    /// This way we can take into acount this offset to move the objects
    /// in qt coords (0,0) at upper left corner
    Vec2i_cu _diff_pixel;

    /// Center of mass of the set '_selection' (in global coordinates)
    Vec3_cu _last_cog;

    /// Center of mass of the set '_selection' (in eye coordinates)
    Vec3_cu _last_cog_eye;

    /// Used to save the state of current data set transformations
    /// in '_selection' attribute
    std::vector<Picker::Data_tr> _saved_transfo_selection;

    /// last saved center of gravity (in global coordinates)
    Vec3_cu _saved_cog;

    /// Axis (x,y,z etc.) used to move the object
    EIO_Selection::Axis_t _axis_mode;

    /// Absolute axis direction set according to '_axis_mode' (global coords)
    Vec3_cu _axis_dir;

    /// User input custom value for transforming objects according to
    /// '_axis_mode' and '_transfo_mode'
    std::list<char> _parsed_value;
    bool _inverse_value;

    /// Settings to draw an infinite line into the scene
    Axis_draw _line;
};

#endif // IO_MANIPULATION_HPP__


