/*
 * This software is governed by the CeCILL-B license under French law and
 * abiding by the rules of distribution of free software.  You can  use, 
 * modify and/ or redistribute the software under the terms of the CeCILL-B
 * license as circulated by CEA, CNRS and INRIA at the following URL
 * "http://www.cecill.info" or the LICENCE.txt file present in this project.
*/

#ifndef IO_SELECTION_HPP__
#define IO_SELECTION_HPP__

#include "IO_selection_enum.hpp"
#include "qt_gui/IO_interface.hpp"
#include "scene_tree.hpp"
#include "scene_renderer.hpp"

#include "obj_hrbf.hpp"

/**
 * @name IO_Selection
 * @brief Generic object selection/manipulation class
 *
 * IO_Selection is dedicated to selecting/removing objects and moving them with
 * the gizmo. More objects movements/manipulation are available within
 * IO_manipulation
 *
 *
 *
 *
 * @see IO_manipulation
 */
class IO_Selection : public IO_interface {
    Q_OBJECT
public:
    // -------------------------------------------------------------------------
    /// @name Events handling
    // -------------------------------------------------------------------------

    IO_Selection(OGL_widget* gl_widget, Scene_renderer* renderer) :
        IO_interface(gl_widget),
        _transfo_mode(EIO_Selection::NONE),
        _is_gizmo_grabed(false),
        _renderer(renderer),
        _gizmo_dir(EIO_Selection::LOCAL),
        _gizmo_org(EIO_Selection::MEDIAN)
    {
        // By default we pick objects of the scene :
        _tree = _renderer->get_scene_tree();
        _current_pick = _renderer->get_picker();
        _obj_pick     = _current_pick;
        _selection    = _current_pick->get_selected();

        _edit_mode = false;
    }

    // -------------------------------------------------------------------------

    virtual ~IO_Selection(){ }

    // -------------------------------------------------------------------------

    void mousePressEvent(QMouseEvent* e)
    {
            if( mouse_press_fall_back(e) ) return;

        const int x = e->x(), y = e->y();
        const Vec2i_cu pos(x, y);

        if(_is_left_pushed)
        {
            // try to select a gizmo axis:
            if( _gl_widget->_draw_gizmo)
                _is_gizmo_grabed = gizmo()->select_constraint(*_cam, x, y);
        }

        if(_is_right_pushed && _transfo_mode == EIO_Selection::NONE)
        {
            // Try to select objects
            bool selected = false;
            if( !_is_gizmo_grabed /*&& !paint */ )
            {
                Data_set s;
                Picker::Params p = { *_cam, ctx()->picker() };
                // selection in the scene
                selected = _current_pick->select(s, pos, p);

                if(_is_shift_pushed)
                    _selection.exclusive_add(s);
                else {
                    if(selected) _selection.clear();
                    _selection.insert(s);
                }

                _current_pick->set_selected_all( false );
                _current_pick->set_selected(_selection, true);

                // Last selected is the active object
                if(s.size() == 1) _current_pick->set_active( *(s.begin())  );
            }
            if(selected) emit_selected_obj();
            // show/hide and orientate the gizmo
            init_gizmo( pos );
        }
    }

    // -------------------------------------------------------------------------

    void mouseReleaseEvent(QMouseEvent* e)
    {
        if( mouse_release_fall_back(e) ) return;

        if(!_is_left_pushed && _is_gizmo_grabed)
        {
            gizmo()->reset_constraint();
            _is_gizmo_grabed = false;
            init_gizmo();

            //if( hrbf data )
            {
                //Transfo tr /**/;
                //_active_pick->transform(_selection, tr);
            }
        }
    }

    // -------------------------------------------------------------------------

    void mouseMoveEvent(QMouseEvent* e)
    {
        if( mouse_move_fall_back(e) ) return;
        const int x = e->x(), y = e->y();
        const Vec2i_cu pos(x, y);

        if( /*not hrbf data && */ _is_left_pushed)
        {
            if(_is_gizmo_grabed)
            {
                TRS gizmo_tr = gizmo()->slide(*_cam, x, y);

                Transfo res = global_transfo( gizmo_tr );

                _current_pick->transform( _selection, res );
                Transfo new_frame = res * gizmo()->frame();
                gizmo()->set_frame( new_frame );
                gizmo()->slide_from( new_frame, pos);
            }

            #if 0
            if( /* paint */ )
                _active_pick->paint( /**/ );
            #endif
        }

    }

    // -------------------------------------------------------------------------

    void wheelEvent(QWheelEvent* e){ if( wheel_fall_back(e) ) return; }

    // -------------------------------------------------------------------------

    void keyPressEvent(QKeyEvent* e)
    {
        if( key_press_fall_back(e) ) return;

        char letter = Qt_utils::event_to_char( e );

        if( _transfo_mode != EIO_Selection::NONE )
            return;

        // SELECT ALL
        if( letter == 'a' )
        {
            if( _selection.size() < 1 ){
                // select all data if nothing is selected
                _current_pick->set_selected_all( true );
                _selection = _current_pick->get_selected();
            } else {
                // unselect if any data is already selected
                _current_pick->set_selected_all( false );
                _selection.clear();
            }
            init_gizmo();
        }
        // CENTER VIEW
        else if( letter == '.' && _enable_cam)
        {
            if( _selection.size() > 0)
                _gl_widget->set_cam_pivot_user_anim( cog(_selection) );
            else
                _gl_widget->set_cam_pivot_user_anim( Vec3_cu::zero() );
        }
        // REMOVE SELECTION
        if( e->key() == Qt::Key_Delete)
        {
            remove_current_selection();
        }
        // RESET TRANSFO
        if( _is_alt_pushed && e->key() == Qt::Key_G )
        {
            reset_transfos();
            init_gizmo();
        }

        //        if(  /* alt r*/ /* alt s */ ) {
        //            // reset rotation
        //        }
#if 0
        // DUPLICATE DATA
        if( /* shift - d*/)
        {
            // Paste selected data
            _copy_buff.copy( _selection );
            _active_pick.add( _copy_buff );
        }

        // APPEND MESH
        if( /* ctrl - j */ )
        {
            // Joins objects into the active object
        }


        if( /* maj - s */ )
        {
            // popup center cursor to selection and all ...
        }

#endif

        if( _is_ctrl_pushed && e->key() == Qt::Key_P){
            Data* active = _current_pick->get_active();
            if( active != 0 && _selection.size() > 0)
                _current_pick->set_parent(_selection, active);
        }

        // edit mode / object mode
        if( e->key() == Qt::Key_Tab )
        {
            #if 1
            if( !_edit_mode )
            {
                Data* d = _current_pick->get_active();
                if( d != 0 && d->type_data() == EData::OBJECT )
                {
                    Obj* o = (Obj*)d;
                    if( o->type_object() == EObj::IMPLICIT ){
                        _current_pick = ((Obj_HRBF*)o)->set_edit_mode();
                        _current_pick->hold();
					}
                }
            }
            else
            {
                _current_pick->release();
                _current_pick = _obj_pick;
//                Data* d = _current_pick->get_active();
//                if( d != 0 && d->type_data() == EData::OBJECT )
//                {
//                    Obj* o = (Obj*)d;
//                    if( o->type_object() == EObj::SKELETON )
//                        ((Obj_skeleton*)o)->set_object_mode();
//                }

            }
            #endif

            _selection = _current_pick->get_selected();
            _edit_mode = !_edit_mode;
        }
    }

    // -------------------------------------------------------------------------

    void keyReleaseEvent(QKeyEvent* e)
    {
        if( key_release_fall_back(e) ) return;
    }

    // -------------------------------------------------------------------------
    /// @name Getter & Setters
    // -------------------------------------------------------------------------

    void set_gizmo_dir(EIO_Selection::Dir_t d){
        _gizmo_dir = d;
        init_gizmo();
    }

    // -------------------------------------------------------------------------

    void set_gizmo_pivot(EIO_Selection::Pivot_t p){
        _gizmo_org = p;
        init_gizmo();
    }

    // -------------------------------------------------------------------------
    /// @name Tools
    // -------------------------------------------------------------------------

    /// @return the current transformation center according to
    /// the mode '_gizmo_org'
    Vec3_cu transfo_center(){ return transfo_frame_from_selection().get_org(); }

    // -------------------------------------------------------------------------

    /// @return the current frame used to transform objects according to the
    /// modes: '_gizmo_org' and '_gizmo_dir'
    Transfo transfo_frame_from_selection()
    {
        using namespace EIO_Selection;
        Data* d = _current_pick->get_active();
        if( d == 0 ) return Transfo::identity();

        Vec3_cu org = Vec3_cu::zero();
        Transfo tr;

        switch(_gizmo_org) {
        case ACTIVE:   { org = d->frame().get_org(); } break;
        case MEDIAN:   { org = cog( _selection );    } break;
        case CURSOR_3D:{ org = get_3d_cursor();      } break;
        }

        // Compute orientation
        switch(_gizmo_dir){
        case GLOBAL: { tr = Transfo::identity();                  } break;
        case LOCAL:  { tr = d->frame();                           } break; // TODO: when editing an object d->frame() is a normal frame and not local -> detect that case
        case NORMAL: { tr = d->frame();                           } break;
        case VIEW:   { tr.set_mat3(_cam->get_frame().get_mat3()); } break; // TODO for this gizmo frame needs to be updated in the viewport
        }

        tr.set_translation( org );
        return tr;
    }

    // -------------------------------------------------------------------------

    /// initialize gizmo position and visibility if acording to the active obj
    void init_gizmo(const Vec2i_cu& pix = Vec2i_cu(-1, -1))
    {
        Data* d = _current_pick->get_active();
        gizmo()->show( d != 0 );
        if( d == 0 ) return;

        Transfo tr = transfo_frame_from_selection();

        gizmo()->set_frame( tr );
        gizmo()->slide_from( tr, pix);
    }

    // -------------------------------------------------------------------------

    Vec3_cu get_3d_cursor() const { return Vec3_cu::zero(); }

signals:
    /// Signal emitted when an active object is selected
    void selected(Obj* o);

private:
    // -------------------------------------------------------------------------
    /// @name Private tools
    // -------------------------------------------------------------------------

    /// Compute center of gravity of a data set
    Vec3_cu cog( const Data_set& set) const
    {
        Vec3_cu v(0.f, 0.f, 0.f);
        Data_set::const_iterator it;
        for(it = set.begin(); it != set.end(); ++it) {
            Data* d = *it;
            v += d->frame().get_org();
        }
        return v / (float)set.size();
    }

    // -------------------------------------------------------------------------

    /// Gizmo global 'TRS' to 4*4 matrix 'Transfo'
    Transfo global_transfo(const TRS& gizmo_tr)
    {
        Transfo tr = Transfo::translate( gizmo_tr._translation );

        Vec3_cu raxis = gizmo_tr._axis;
        Vec3_cu org = gizmo()->frame().get_org();
        Transfo rot = Transfo::rotate(org, raxis, gizmo_tr._angle);
        Transfo sc  =  Transfo::scale(org, gizmo_tr._scale );

        return tr * rot * sc;
    }

    // -------------------------------------------------------------------------

    /// emit 'void selected(Obj* o)' is there is an active object
    void emit_selected_obj()
    {
        Data* d = _current_pick->get_active();
        if( d != 0 && d->type_data() == EData::OBJECT) {
            emit selected( (Obj*)d );
        }
    }

    // -------------------------------------------------------------------------

    /// Sets local transformations of every selected objs to indentity
    void reset_transfos()
    {
        std::vector<Picker::Data_tr> tmp( _selection.size() );
        Data_set::iterator it = _selection.begin();
        for(int i = 0; it != _selection.end(); ++it, ++i)
            tmp[i] = std::make_pair(*it, Transfo::identity());

        _current_pick->load_transform( tmp );
    }

    // -------------------------------------------------------------------------

    /// Remove every objects currently selected
    void remove_current_selection()
    {
        _current_pick->set_selected( _selection, false);
        _current_pick->remove( _selection );
        _selection.clear();
        gizmo()->show( false );
    }

    // -------------------------------------------------------------------------
    /// @name IO fall backs
    // -------------------------------------------------------------------------

    bool mouse_press_fall_back(QMouseEvent* e)
    {
        Picker_io* spe_io = _current_pick->get_io_extension();
        if( spe_io != 0 && (spe_io->_io_selection = this) && spe_io->mouse_press( e ) )
                return true;
        IO_interface::mousePressEvent(e);
        return false;
    }

    // -------------------------------------------------------------------------

    bool mouse_release_fall_back(QMouseEvent* e)
    {
        Picker_io* spe_io = _current_pick->get_io_extension();
        if( spe_io != 0 && (spe_io->_io_selection = this) && spe_io->mouse_release( e ) )
                return true;
        IO_interface::mouseReleaseEvent(e);
        return false;
    }

    // -------------------------------------------------------------------------

    bool mouse_move_fall_back(QMouseEvent* e)
    {
        Picker_io* spe_io = _current_pick->get_io_extension();
        if( spe_io != 0 && (spe_io->_io_selection = this) && spe_io->mouse_move( e ) )
                return true;
        IO_interface::mouseMoveEvent(e);
        return false;
    }

    // -------------------------------------------------------------------------

    bool wheel_fall_back(QWheelEvent* e)
    {
        Picker_io* spe_io = _current_pick->get_io_extension();
        if( spe_io != 0 && (spe_io->_io_selection = this) && spe_io->wheel( e ) )
                return true;
        IO_interface::wheelEvent(e);
        return false;
    }

    // -------------------------------------------------------------------------

    bool key_press_fall_back(QKeyEvent* e)
    {
        Picker_io* spe_io = _current_pick->get_io_extension();
        if( spe_io != 0 && (spe_io->_io_selection = this) && spe_io->key_press( e ) )
                return true;
        IO_interface::keyPressEvent(e);
        return false;
    }

    // -------------------------------------------------------------------------

    bool key_release_fall_back(QKeyEvent* e)
    {
        Picker_io* spe_io = _current_pick->get_io_extension();
        if( spe_io != 0 && (spe_io->_io_selection = this) && spe_io->key_release( e ) )
                return true;
        IO_interface::keyReleaseEvent(e);
        return false;
    }

protected:
    // -------------------------------------------------------------------------
    /// @name Attributes
    // -------------------------------------------------------------------------
    /// Current manipulation mode. If not set to NONE selection is disabled
    EIO_Selection::Transfo_t _transfo_mode;

    Picker*  _current_pick; ///< current object/data picker
    Data_set _selection;    ///< current selection set

private:
    bool _is_gizmo_grabed;     ///< is a gizmo constraint selected

    bool _edit_mode;

    Picker* _obj_pick;         ///< Object Scene picker

    Data_set_copy _copy_buff;  ///< buffer to copy datas

    Scene_tree*     _tree;
    Scene_renderer* _renderer;

    /// Gizmo orientation mode (local, global...)
    EIO_Selection::Dir_t   _gizmo_dir;
    /// Gizmo origin mode (Median, 3D cursor...)
    EIO_Selection::Pivot_t _gizmo_org;
};

#endif // IO_SELECTION_HPP__
