/*
 * This software is governed by the CeCILL-B license under French law and
 * abiding by the rules of distribution of free software.  You can  use, 
 * modify and/ or redistribute the software under the terms of the CeCILL-B
 * license as circulated by CEA, CNRS and INRIA at the following URL
 * "http://www.cecill.info" or the LICENCE.txt file present in this project.
*/

#ifndef OGL_VIEWPORTS_SKIN2_HPP__
#define OGL_VIEWPORTS_SKIN2_HPP__

#include <QFrame>
#include <vector>
#include <QLayout>

#include "main_window.hpp"
#include "OGL_widget.hpp"


class Viewport_frame;

typedef  std::vector<OGL_widget*> Vec_viewports;

// =============================================================================
class OGL_widget_hidden : public QGLWidget {
public:
    OGL_widget_hidden(QWidget* w) : QGLWidget(w) {}

    void initializeGL();
};
// =============================================================================

/** @class OGL_viewports
  @brief Multiple viewports handling with different layouts

*/
class OGL_viewports : public QFrame {
    Q_OBJECT
public:
    OGL_viewports(QWidget* w, Main_window* m);

    /// Updates all viewports
    void updateGL();

    // -------------------------------------------------------------------------
    /// @name Getter & Setters
    // -------------------------------------------------------------------------
    enum Layout_e { SINGLE, VDOUBLE, HDOUBLE, FOUR };

    /// Erase all viewports and rebuild them according to the specified layout
    /// 'setting'
    void set_viewports_layout(Layout_e setting);

    /// @warning the list is undefined if used after a call to
    /// set_viewports_layout()
    Vec_viewports& get_viewports();

    /// sets gizmo type (rotation, scaling, translation) for all viewports
    void set_gizmo(Gizmo::Gizmo_t type);

    /// Show/hide gizmo for all viewports
    void show_gizmo(bool state);

    /// sets camera pivot for all viewports
    void set_cam_pivot(EOGL_widget::Pivot_t m);

    /// sets gizmo pivot for all viewports
    void set_gizmo_pivot(EIO_Selection::Pivot_t piv);

    /// sets gizmo orientation for all viewports
    void set_gizmo_dir(EIO_Selection::Dir_t dir);

    void set_alpha_strength(float a);

    /// @return the active frame
    OGL_widget* active_viewport(){ return _current_viewport; }

    QGLWidget* shared_viewport(){ return _hidden; }

    // -------------------------------------------------------------------------
    /// @name Events
    // -------------------------------------------------------------------------
    void enterEvent( QEvent* e);

    // -------------------------------------------------------------------------
    /// @name Qt Signals & Slots
    // -------------------------------------------------------------------------
private slots:
    /// Designed to be called each time a single viewport draws one frame.
    void incr_frame_count();
    void active_viewport_slot(int id);

signals:
    void frame_count_changed(int);
    void active_viewport_changed(int id);
    /// Update status bar
    void update_status(QString);

private:
    // -------------------------------------------------------------------------
    /// @name Tools
    // -------------------------------------------------------------------------
    QLayout* gen_single ();
    QLayout* gen_vdouble();
    QLayout* gen_hdouble();
    QLayout* gen_four   ();

    /// suppress all viewports and layouts
    void erase_viewports();

    /// Creates a new viewport with the correct signals slots connections
    OGL_widget* new_viewport(Viewport_frame* ogl_frame);

    /// Creates a new viewport frame with the correct signals slots connections
    Viewport_frame* new_viewport_frame(QWidget* parent, int id);

    /// Sets the frame color by replacing its styleSheet color
    void set_frame_border_color(Viewport_frame* f, int r, int g, int b);

    void first_viewport_as_active();

    /// Before drawing compute every pre process like mesh deformation
    void update_scene();

    // -------------------------------------------------------------------------
    /// @name Attributes
    // -------------------------------------------------------------------------

    /// Vector of OGL_widget
    Vec_viewports _viewports;

    /// List of frames associated to the viewports
    std::vector<Viewport_frame*> _viewports_frame;

    /// The active viewport
    OGL_widget* _current_viewport;

    /// opengl shared context between all viewports
    /// (in order to share VBO textures etc.)
    OGL_widget_hidden* _hidden;

    /// Layout containing all viewports
    QLayout* _main_layout;

    /// main widow the widget's belongs to
    Main_window* _main_window;

    /// sum of frames drawn by the viewports
    int _frame_count;

    /// Fps counting timer
    QTime  _fps_timer;
};
// =============================================================================

class Viewport_frame : public QFrame {
    Q_OBJECT
public:

    Viewport_frame(QWidget* w, int id) : QFrame(w), _id(id)
    {
        setFrameShape(QFrame::Box);
        setFrameShadow(QFrame::Plain);
        setLineWidth(1);
        setStyleSheet(QString::fromUtf8("color: rgb(0, 0, 0);"));
    }


    int id() const { return _id; }

signals:
    void active(int);

private slots:
    void activate(){
        emit active(_id);
    }

private:
    int _id;
};
// =============================================================================

#endif // OGL_VIEWPORTS_SKIN2_HPP__
