/*
 * This software is governed by the CeCILL-B license under French law and
 * abiding by the rules of distribution of free software.  You can  use, 
 * modify and/ or redistribute the software under the terms of the CeCILL-B
 * license as circulated by CEA, CNRS and INRIA at the following URL
 * "http://www.cecill.info" or the LICENCE.txt file present in this project.
*/

#include "OGL_widget.hpp"

#include <QMouseEvent>
#include <QKeyEvent>
#include <iostream>

#include "main_window.hpp"
#include "IO_disable.hpp"
#include "IO_manipulation.hpp"

#include "opengl_stuff.hpp"
#include "common/gizmo/gizmo_trans.hpp"
#include "common/gizmo/gizmo_rot.hpp"
#include "common/gizmo/gizmo_trackball.hpp"
#include "common/gizmo/gizmo_scale.hpp"

//#include <QtOpenGL>
#include <limits>

#include "g_scene_tree.hpp"
// From globals.hpp
extern bool g_shooting_state;
extern bool g_save_anim;

// -----------------------------------------------------------------------------

void OGL_widget::init_glew()
{
    glewInit();
    int state = glewIsSupported("GL_VERSION_2_0 "
                                "GL_VERSION_1_5 "
                                "GL_ARB_vertex_buffer_object "
                                "GL_ARB_pixel_buffer_object");
    if(!state) {
        fprintf(stderr, "Cannot initialize glew: required OpenGL extensions missing.");
        exit(-1);
    }
}

// -----------------------------------------------------------------------------

void OGL_widget::init()
{
    _pivot_user = Vec3_cu(0.f, 0.f, 0.f);
    _pivot      = Vec3_cu(0.f, 0.f, 0.f);
    _cam_pivot  = EOGL_widget::USER;

    _draw_gizmo  = true;
    _track_pivot = false;

    _io_disabled  = new IO_disable(this);
    _io_manipulation = 0; // Must be initialized after the renderer
    _io           = _io_disabled;

    _msge_stack  = new Msge_stack(this);
    _heuristic   = new Selection_nearest<int>();
    _renderer    = 0; // Must be initialized after OpenGL context setup
    _gizmo       = 0; // idem ...

    _refresh_screen_timer = new QTimer(this);
    connect(_refresh_screen_timer, SIGNAL(timeout()), this, SLOT(updateGL()));
    connect(&_anim_track_pivot_timer, SIGNAL(timeout()), this, SLOT(anim_track_pivot()));
    connect(_msge_stack, SIGNAL(updateGL()), this, SLOT(updateGL()));
}

// -----------------------------------------------------------------------------

OGL_widget::OGL_widget(QWidget *parent, Main_window* m) :
    QGLWidget(parent),
    _main_win(m)
{
    this->init();
}

// -----------------------------------------------------------------------------

OGL_widget::OGL_widget(QWidget *parent,  QGLWidget* sh, Main_window* m):
    QGLWidget(parent, sh),
    _main_win(m)
{
    this->init();
}

// -----------------------------------------------------------------------------

OGL_widget::~OGL_widget()
{
    makeCurrent();
    delete _heuristic;
    delete _refresh_screen_timer;
    delete _msge_stack;
    delete _renderer;
    delete _gizmo;
    delete _io_disabled;
    delete _io_manipulation;
    _io = 0;
}

// -----------------------------------------------------------------------------

void OGL_widget::initializeGL()
{
    makeCurrent();
    OGL_widget::init_glew();

    _cam.set_pos( Vec3_cu(-70.f, -90.f, 50.f) );
    _cam.lookat( Vec3_cu::zero() );
    _cam.set_viewport(0, 0, width(), height());

    // Make 'up' as close as possible to 'z'
    Vec3_cu dir = _cam.get_dir().normalized();
    Vec3_cu axe = Vec3_cu::unit_z();
    _cam.set_dir_and_up( dir, (dir.cross( axe )).cross( dir ) );

    _renderer = new Scene_renderer(&_cam, g_scene_tree, width(), height());
    _gizmo    = new Gizmo_trans( ctx()->picker() );
    _io_manipulation = new IO_Manipulation(this, _renderer);
    _io = _io_manipulation;
    connect( _io_manipulation, SIGNAL(selected(Obj*)), this, SLOT(set_selected(Obj*)) );
}

// -----------------------------------------------------------------------------

void OGL_widget::resizeGL(int w, int h) {
    makeCurrent();
    glViewport(0, 0, w, h);
    _renderer->render_context()->reshape(w, h);
    _cam.set_viewport(0, 0, w, h);
}

// -----------------------------------------------------------------------------

void OGL_widget::paintGL() {
    makeCurrent(); // TODO: this call might be useless check doc

    // Correct camera position if tracking is on
    update_pivot();
    update_camera();

    // Render the scene
    bool need_to_refresh = _renderer->draw();

    // DEBUG to be deleted **************************************************************

    // DEBUG to be deleted **************************************************************

    if(g_save_anim || g_shooting_state)
        emit drawing();

    draw_grid_lines( &_cam );

    if( need_to_refresh )
    {
        _refresh_screen_timer->setSingleShot(true);
        _refresh_screen_timer->start(1);
    }

    if(_heuristic->_type == Selection::CIRCLE && _is_mouse_in)
    {
        Selection_circle<int>* h = (Selection_circle<int>*)_heuristic;
        glColor3f(0.f, 0.f, 0.f);
        draw_circle(_cam.width(), _cam.height(), (float)_mouse_x, (float)_mouse_y, h->_rad);
    }

    _io_manipulation->draw_manipulator();

    if(_draw_gizmo) _gizmo->draw(_cam);


    // Draw latest text messages
    Color cl = Color::black();
    glColor4f(cl.r, cl.g, cl.b, cl.a);
    _msge_stack->draw(5, this->height()-35);
}

// -----------------------------------------------------------------------------

void OGL_widget::set_gizmo(Gizmo::Gizmo_t type)
{
    makeCurrent();
    Gizmo* tmp = _gizmo;
    switch(type)
    {
    case Gizmo::TRANSLATION: _gizmo = new Gizmo_trans( ctx()->picker() ); break;
    case Gizmo::SCALE:       _gizmo = new Gizmo_scale();                  break;
    case Gizmo::ROTATION:    _gizmo = new Gizmo_rot();                    break;
    case Gizmo::TRACKBALL:   _gizmo = new Gizmo_trackball();              break;
    }
    _gizmo->copy(tmp);
    delete tmp;
    updateGL();
}

// -----------------------------------------------------------------------------

void OGL_widget::set_selection(EOGL_widget::Select_t select_mode)
{
    delete _heuristic;
    switch(select_mode)
    {
    case EOGL_widget::MOUSE:     _heuristic = new Selection_nearest<int>(); break;
    case EOGL_widget::CIRCLE:    _heuristic = new Selection_circle<int>();  break;
    case EOGL_widget::BOX:       _heuristic = 0; break;
    case EOGL_widget::FREE_FORM: _heuristic = 0; break;
    default:        _heuristic = 0; break;
    }
}

// -----------------------------------------------------------------------------

void OGL_widget::set_selected(Obj* o){
    emit selected(o);
}

// -----------------------------------------------------------------------------

void OGL_widget::anim_track_pivot()
{
    static bool  anim_on   = false;
    static float step_size = std::numeric_limits<float>::infinity();

    // Camera position must translate to 'dst'
    Vec3_cu dst = _pivot_user - _cam.get_dir() * 5.f;
    // Anim follow the line defined by 'dir_anim'
    Vec3_cu dir_anim = dst - _cam.get_pos();

    // What's left to go is equal to norm
    float norm = dir_anim.safe_normalize();

    // If the anim begin (i.e: !anim_on) we travel by steps equal to 'step_size'
    // norm must be greater than zero otherwise we don't need to animate
    if( !anim_on && norm > 0.00001f ){
        step_size = norm / 10.f;
        anim_on = true;
    }

    if( step_size > norm )
    {
        // step_size is smaller than the distance to travel -> stop anim and
        // go directly to dst
        _cam.set_pos( dst );
        _anim_track_pivot_timer.stop();
        anim_on = false;
        step_size = std::numeric_limits<float>::infinity();
    }
    else
    {
        // Proceed one step of animation
        dst = _cam.get_pos() + dir_anim * step_size;
        _cam.set_pos( dst );
    }

    updateGL();
}

// -----------------------------------------------------------------------------

void OGL_widget::set_main_window(Main_window* m)
{
    _main_win = m;
}

// -----------------------------------------------------------------------------

Main_window* OGL_widget::get_main_window()
{
    return _main_win;
}

// -----------------------------------------------------------------------------

void OGL_widget::mousePressEvent( QMouseEvent* event ){
    makeCurrent();
    emit clicked();
    _io->mousePressEvent(event);
    _main_win->update_viewports();
}

// -----------------------------------------------------------------------------

void OGL_widget::mouseReleaseEvent( QMouseEvent* event ){
    makeCurrent();
    _io->mouseReleaseEvent(event);
    _main_win->update_viewports();
}

// -----------------------------------------------------------------------------

void OGL_widget::wheelEvent( QWheelEvent* event ){
    makeCurrent();
    _io->wheelEvent(event);
    _main_win->update_viewports();
    event->accept();
}

// -----------------------------------------------------------------------------

void OGL_widget::mouseMoveEvent( QMouseEvent* event ){
    makeCurrent();
    _mouse_x = event->x();
    _mouse_y = event->y();

    _io->mouseMoveEvent(event);

    _main_win->update_viewports();
}

// -----------------------------------------------------------------------------

void OGL_widget::keyPressEvent( QKeyEvent* event ){
    makeCurrent();
    _io->keyPressEvent(event);
    if( !event->isAutoRepeat() )
        _main_win->update_viewports();
}

// -----------------------------------------------------------------------------

void OGL_widget::keyReleaseEvent( QKeyEvent* event ){
    makeCurrent();
    _io->keyReleaseEvent(event);
    _main_win->update_viewports();
}

// -----------------------------------------------------------------------------

void OGL_widget::enterEvent( QEvent* e){
    _is_mouse_in = true;
    _main_win->update_viewports();
    e->ignore();
}

// -----------------------------------------------------------------------------

void OGL_widget::leaveEvent( QEvent* e){
    _is_mouse_in = false;
    _main_win->update_viewports();
    e->ignore();
}

// -----------------------------------------------------------------------------

bool OGL_widget::event(QEvent *myEvent)
{
    bool accepted = false;
    QEvent::Type type = myEvent->type();
    if(type == QEvent::KeyPress || type == QEvent::KeyRelease)
    {
        QKeyEvent* keyEvent = dynamic_cast<QKeyEvent*>(myEvent);
        int key = keyEvent->key();
        if( key ==  Qt::Key_Tab ) accepted = true;

        if(accepted){
            if(type == QEvent::KeyPress) keyPressEvent(keyEvent);
            else                         keyReleaseEvent(keyEvent);

            myEvent->accept();
            return true;
        }
    }

    return QGLWidget::event(myEvent);
}

// -----------------------------------------------------------------------------

void OGL_widget::set_gizmo_pivot( EIO_Selection::Pivot_t piv)
{
    makeCurrent();
    _io_manipulation->set_gizmo_pivot(piv);
    updateGL();
}

// -----------------------------------------------------------------------------

void OGL_widget::set_gizmo_dir( EIO_Selection::Dir_t dir)
{
    makeCurrent();
    _io_manipulation->set_gizmo_dir(dir);
    updateGL();
}

// -----------------------------------------------------------------------------

void OGL_widget::set_cam_pivot_user(const Vec3_cu& v){
    _pivot_user = v;
    updateGL();
}

// -----------------------------------------------------------------------------

void OGL_widget::set_cam_pivot_user_anim(const Vec3_cu& v)
{
    // Check if anim is not already running
    if( _anim_track_pivot_timer.isActive() ) return;

    _pivot_user = v;
    // Launch anim at 33,33 fps.
    // Timer will call back the animation slot: 'anim_track_pivot()'
    _anim_track_pivot_timer.start(30);
}

// -----------------------------------------------------------------------------

void OGL_widget::update_pivot()
{


    if(_cam_pivot == EOGL_widget::FREE)
        return;
    else if(_cam_pivot == EOGL_widget::SELECTION)
    {
        //DEPRECATED
    }
    else if(_cam_pivot == EOGL_widget::USER)
            _pivot = _pivot_user;
    else if(_cam_pivot == EOGL_widget::BONE)
    {
        //DEPRECATED
    }
    else if(_cam_pivot == EOGL_widget::JOINT)
    {
        //DEPRECATED
    }
}

// -----------------------------------------------------------------------------

void OGL_widget::update_camera()
{
    if(!_track_pivot) return;

    Vec3_cu ndir = _pivot - _cam.get_pos();
    float dist   = ndir.normalize();
    Vec3_cu up   = _cam.get_y().normalized();
    Vec3_cu x    = (up.cross(ndir)).normalized();

    Vec3_cu nup = (ndir.cross(x)).normalized();

    if(_cam.get_far()*0.9 < dist)
        _cam.set_far( dist + dist*0.1);

    _cam.set_dir_and_up(ndir, nup);
}


// -----------------------------------------------------------------------------
