/*
 * This software is governed by the CeCILL-B license under French law and
 * abiding by the rules of distribution of free software.  You can  use, 
 * modify and/ or redistribute the software under the terms of the CeCILL-B
 * license as circulated by CEA, CNRS and INRIA at the following URL
 * "http://www.cecill.info" or the LICENCE.txt file present in this project.
*/

#ifndef _OGL_WIDGET_SKIN2_H_
#define _OGL_WIDGET_SKIN2_H_

#include <QEvent>
#include <QKeyEvent>
#include <QTimer>
#include <QTime>

#include "selection_heuristic.hpp"
#include "portable_includes/port_glew.h"
#include "camera.hpp"
#include "common/gizmo/gizmo.hpp"

#include "common/msge_stack.hpp"
#include <QGLWidget>
#include "common/OGL_widget_enum.hpp"
#include "IO_selection_enum.hpp"

#include "scene_renderer.hpp"

// FORWARD DEFS ----------------------------------------------------------------
// Can't include the header because of interdependencies between IO_interface
// and OGLWidget classes
class IO_interface;
class IO_disable;
class IO_Manipulation;
class Main_window;
// END FORWARD DEFS ------------------------------------------------------------

// =============================================================================

class OGL_widget : public QGLWidget {
    Q_OBJECT
public:

    // -------------------------------------------------------------------------
    /// @name Constructors
    // -------------------------------------------------------------------------

    OGL_widget(QWidget *parent, QGLWidget* sh, Main_window* m);
    OGL_widget(QWidget *parent, Main_window* m);
    ~OGL_widget();

    /// update the pivot position given the current mode of rotation
    void update_pivot();

    // -------------------------------------------------------------------------
    /// @name Static tools
    // -------------------------------------------------------------------------

    /// initialize the glew to access openGL extensions must be done once per
    /// openGL context
    static void init_glew();

    // -------------------------------------------------------------------------
    /// @name Getter & Setters
    // -------------------------------------------------------------------------

          Render_context* ctx()       { return _renderer->render_context(); }
    const Render_context* ctx() const { return _renderer->render_context(); }

    void set_alpha_strength(float a){ _renderer->set_alpha_strength(a); }

    Select_type<int>* get_heuristic(){ return _heuristic; }

    void set_main_window(Main_window* m);
    Main_window* get_main_window();

    // -------------------------------------------------------------------------
    /// @name Gizmo accessors
    // -------------------------------------------------------------------------

    const Gizmo* gizmo() const { return _gizmo; }
          Gizmo* gizmo()       { return _gizmo; }

    /// Choose the type of gizmo (rotation translation scale). Origin and
    /// orientation are kept fropm the old gizmo
    void set_gizmo(Gizmo::Gizmo_t type);

    void set_gizmo_pivot( EIO_Selection::Pivot_t piv);

    void set_gizmo_dir( EIO_Selection::Dir_t dir);

    // -------------------------------------------------------------------------
    /// @name Camera accessors
    // -------------------------------------------------------------------------

    Camera* camera(){ return &_cam; }

    /// Get the current pivoting point the camera use.
    Vec3_cu cam_pivot_pos() const { return _pivot; }

    EOGL_widget::Pivot_t cam_pivot_type() const { return _cam_pivot; }

    void set_cam_pivot_type(EOGL_widget::Pivot_t m){ _cam_pivot = m; }

    /// Set the pivot point in user mode without animation
    void set_cam_pivot_user(const Vec3_cu& v);

    /// Set the pivot point in user mode with animation
    void set_cam_pivot_user_anim(const Vec3_cu& v);

    // -------------------------------------------------------------------------
    /// @name Public attributes
    // -------------------------------------------------------------------------

    /// use to draw temporary message on screen
    Msge_stack* _msge_stack;

    /// wether we draw and use the for objects movements
    bool _draw_gizmo;

    /// wether the camera tracks the pivot point or not
    bool _track_pivot;

signals:
    // -------------------------------------------------------------------------
    /// @name Signals
    // -------------------------------------------------------------------------

    /// Emited on mouse press events
    void clicked();
    /// Emited for each frame (only if (g_save_anim || g_shooting_state))
    void drawing();
    /// Emited when a new active object is selected
    void selected(Obj*);

public slots:
    // -------------------------------------------------------------------------
    /// @name Slots
    // -------------------------------------------------------------------------

    /// set the selection mode
    void set_selection(EOGL_widget::Select_t select_mode);

    /// Sets the active selected object.
    void set_selected(Obj* o);

    /// Do one step of animation to track the pivot
    void anim_track_pivot();

 protected:
    // -------------------------------------------------------------------------
    /// @name Events
    // -------------------------------------------------------------------------

    void initializeGL();
    void resizeGL(int w, int h);
    void paintGL();
    void mousePressEvent  ( QMouseEvent* event );
    void mouseReleaseEvent( QMouseEvent* event );
    void wheelEvent       ( QWheelEvent* event );
    void mouseMoveEvent   ( QMouseEvent* event );
    void keyPressEvent    ( QKeyEvent*   event );
    void keyReleaseEvent  ( QKeyEvent*   event );
    void enterEvent       ( QEvent*      event );
    void leaveEvent       ( QEvent*      event );

    /// Hack to generate a key event when tab is stroke.
    /// By default Qt receive the event but eat it in order to switch the focus
    /// from widget to widget. We bypass this behavior by overriding the event()
    /// method. This enables us to forward the tab event to keyPressEvent() or
    /// keyReleaseEvent()
    bool event( QEvent* event );

private:

    // -------------------------------------------------------------------------
    /// @name Internal tools
    // -------------------------------------------------------------------------

    /// Factorize the initialization of some attributes
    void init();

    /// if necessary updates camera position when tracking enabled
    void update_camera();

    // -------------------------------------------------------------------------
    /// @name Attributes
    // -------------------------------------------------------------------------

    /// @defgroup input/output handlers
    /// @{
    /// handle mouse and keyboards according to the desire modeling mode
    IO_interface* _io;
    IO_disable*   _io_disabled;
    IO_Manipulation*    _io_manipulation;
    /// @{

    /// The main window this viewport belongs to.
    Main_window* _main_win;

    /// Context and rendering algorithm for cuda and openGL
    Scene_renderer* _renderer;

    /// Camera attached to the viewport
    Camera _cam;

    /// Pivot point position. Used by the camera to rotate around it.
    Vec3_cu _pivot;

    /// Pivot point defined by the user
    Vec3_cu _pivot_user;

    /// mode of rotation: defines what's the standard behavior to compute
    /// automatically the pivot point
    EOGL_widget::Pivot_t _cam_pivot;

    /// 3d manipulator to move objects of the scene
    Gizmo* _gizmo;

    /// Current heuristic for mesh's points selection. Which defines the
    /// selection area (point, square, circle etc.) used to select the mesh's
    /// points.
    Select_type<int>* _heuristic;

    /// use to redraw screen at regular intervalles
    QTimer* _refresh_screen_timer;

    /// Used to do to the pivot tracking animation
    QTimer _anim_track_pivot_timer;

    /// true when mouse cursor hoover the viewport
    bool _is_mouse_in;

    /// Current mouse position
    int _mouse_x, _mouse_y;
};

#endif // _OGL_WIDGET2_H_
