/*
 * This software is governed by the CeCILL-B license under French law and
 * abiding by the rules of distribution of free software.  You can  use, 
 * modify and/ or redistribute the software under the terms of the CeCILL-B
 * license as circulated by CEA, CNRS and INRIA at the following URL
 * "http://www.cecill.info" or the LICENCE.txt file present in this project.
*/

#ifndef GIZMO2_HPP__
#define GIZMO2_HPP__

#include "trs.hpp"
#include "vec3_cu.hpp"
#include "vec2i_cu.hpp"
#include "camera.hpp"
#include "glpick_fbo.hpp"

/**
  @class Gizmo
  @brief a Gizmo is a 3D object enabling the user to transform an a object of
  the scene

  This base class is used to represent a 3D object: the Gizmo. It can be
  displayed and grabbed with the mouse to enable the user moving objects into
  the scene.

  Usually this class is intended to be specialized to perform translation,
  rotation and scaling.

  One can set the gizmo global orientation and position in space using :
  @code
  set_frame();
  set_transfo();
  set_org();
  @endcode

  the method 'draw()' paint with openGL the gizmo given it's
  position/orientation (i.e attribute '_frame'). Hidding the gizmo and disabling
  its selection is done with show(bool state).

  Use case (incremental version):
  @code
  // Instantiate a gizmo to rotate
  Gizmo* gizmo = new Gizmo_rot();

  void mouse_click_on_pixel(int x, int y)
  {
      if( gizmo->select_constraint(cam, x, y) )
      {
            // Constraint has been selected
            gizmo->set_frame( selection_transfo ); // Set drawing postion
            gizmo->show(true); // enable drawing
            gizmo->slide_from( tr, pix);
      }
  }

  void mouse_move_on_pixel(int x, int y)
  {
      // Get the transformation of the gizmo.
      // (expressed in global coordinates
      TRS gizmo_tr = gizmo->slide(cam, x, y);
      Transfo res = global_transfo( gizmo_tr );
      apply_transfo( res );
      Transfo new_frame = res * gizmo->frame();
      gizmo->set_frame( new_frame );
      gizmo->slide_from( new_frame, pos);
  }

  void mouse_release(){
      gizmo->reset_constraint();
  }

  void paintGlLoop(){
      gizmo->draw(cam);
  }
  @endcode

  @see Gizmo_scale Gizmo_rot Gizmo_trans Gizmo_trackball
*/
class Gizmo {
public:
    enum Gizmo_t { TRANSLATION, ROTATION, TRACKBALL, SCALE };

    // =========================================================================
    /// @name Constructors
    // =========================================================================

    Gizmo() :
        _frame( Transfo::identity() ),
        _show(false),
        _picker(0)
    { }

    Gizmo(GlPick_FBO* picker) :
        _frame( Transfo::identity() ),
        _show(false),
        _picker( picker )
    { }

    virtual ~Gizmo(){ }

    void copy(const Gizmo* obj){
        _frame = obj->_frame;
        _show  = obj->_show;
    }

    // =========================================================================
    /// @name Getters & Setters
    // =========================================================================

    /// Set frame orientation and origin (world coordinates) used
    /// to draw the gizmo and its selection
    /// @see set_org() set_frame()
    void set_frame(const Transfo& tr) { _frame = tr.normalized(); }

    /// @return the current frame of the gizmo used to draw it
    Transfo frame() const { return _frame; }

    // =========================================================================
    /// @name Drawing
    // =========================================================================

    /// Draw the gizmo according to its orientation and origin
    /// (_frame attribute).
    /// Selected constraint with 'select_constraint()' will be highlighted.
    /// drawing must be enabled with show(true)
    /// @see set_transfo() show() select_constraint() reset_constraint()
    virtual void draw(const Camera& cam) = 0;

    /// Disable the gizmo drawing and selection
    void show(bool state){ _show = state; }

    // =========================================================================
    /// @name Gizmo selection
    // =========================================================================

    /// select a constraint (for instance the 'x' axis of translation)
    /// given a camera and a mouse position.
    /// this calls slide_from( _frame, Vec2i_cu(px, py) )
    /// @return true if a constraint has been selected
    virtual bool select_constraint(const Camera& cam, int px, int py) = 0;

    /// reset the selected constraint set by select_constraint(),
    /// this disable slide and the highlighting with draw
    virtual void reset_constraint() = 0;

    // =========================================================================
    /// @name Compute transformation
    // =========================================================================

    /// Sets starting position to compute the slide
    void slide_from( const Transfo start_frame, const Vec2i_cu& start_pix){
        _start_frame = start_frame;
        _start_pix   = start_pix;
    }

    /// @brief slide the gizmo along the current selected constraint.
    /// Given a new mouse position (px, py) we deduce the transformation made
    /// by the gizmo knowing the selected constraint (c.f. select_constraint())
    /// and the old mouse position and old orientation (c.f. slide_from()).
    /// We try to keep the mouse as close as possible under the gizmo
    /// @note to change the drawn position of the gizmo when calling 'draw()'
    /// don't forget to update the transformation with a set_frame()
    /// @return The transformation made by the gizmo in global coordinates
    /// @warning
    /// @see select_constraint() reset_constraint() set_transfo()
    virtual TRS slide(const Camera& cam, int px, int py) = 0;

protected:
    Transfo  _start_frame;
    Vec2i_cu _start_pix;

    Transfo _frame; ///< orientation and position of the gizmo
    bool    _show;  ///< do we draw the gizmo

    GlPick_FBO* _picker; ///< Utility to select opengl objects
};


#endif // GIZMO2_HPP__
