/*
 * This software is governed by the CeCILL-B license under French law and
 * abiding by the rules of distribution of free software.  You can  use, 
 * modify and/ or redistribute the software under the terms of the CeCILL-B
 * license as circulated by CEA, CNRS and INRIA at the following URL
 * "http://www.cecill.info" or the LICENCE.txt file present in this project.
*/

#include "common/gizmo/gizmo_trans.hpp"

#include <limits>

#include "glsave.hpp"
#include "intersection.hpp"
#include "vec3_cu.hpp"
#include "vec2_cu.hpp"
#include "conversions.hpp"



// -----------------------------------------------------------------------------

void Gizmo_trans::init_attr()
{
    _size = 10.f;
    _length = 0.13f;
    _radius = 0.30f/50.f;
    _ortho_factor = 1.f;
    _selected_axis = NOAXIS;
    _axis = Vec3_cu::unit_x();
}

// -----------------------------------------------------------------------------

Gizmo_trans::Gizmo_trans(GlPick_FBO* picker): Gizmo( picker )
{
    init_attr();
}

// -----------------------------------------------------------------------------

static void draw_arrow(float radius, float length)
{
    GLUquadricObj* quad = gluNewQuadric();
    glPushMatrix();
        glBegin(GL_LINES);
            glVertex3f(0.f, 0.f, 0.f);
            glVertex3f(0.f, 0.f, 0.8f*length);
        glAssert( glEnd() );

        glTranslatef(0.f, 0.f, 0.8f * length);
        gluCylinder(quad, radius, 0.0f  , 0.2f * length, 10, 10);
    glPopMatrix();

    gluDeleteQuadric(quad);
}

// -----------------------------------------------------------------------------

static void draw_quad( float l )
{
    l *= 0.2f;

    glBegin(GL_QUADS);
        glVertex3f(0.f, 0.f, 0.f);
        glVertex3f(l  , 0.f, 0.f);
        glVertex3f(l  , l, 0.f);
        glVertex3f(0.f , l, 0.f);
    glEnd();

    GLPolygonModeSave mode( GL_LINE );
    GLLineWidthSave width(1.0f);
    glColor4f( 0.3f, 0.3f, 0.3f, 1.f);
    glBegin(GL_LINES);
        glVertex3f(l  , 0.f, 0.f);
        glVertex3f(l  , l, 0.f);
        glVertex3f(l  , l, 0.f);
        glVertex3f(0.f , l, 0.f);
    glEnd();
}

// -----------------------------------------------------------------------------

void Gizmo_trans::draw_quads()
{
    glPushMatrix();
    {
        /* PLANE XY */
        if(_selected_axis == XY )
            glColor4f(1.f, 1.f, 0.f, 0.5f);
        else
            glColor4f( 0.4f, 0.4f, 0.4f, 0.5f);

        if( _picker->is_pick_init() ) _picker->set_name( XY );
        draw_quad( _length );

        /* PLANE XZ */
        if(_selected_axis == XZ )
            glColor4f(1.f, 1.f, 0.f, 0.5f);
        else
            glColor4f( 0.4f, 0.4f, 0.4f, 0.5f);

        glRotatef(90.f, 1.f, 0.f, 0.f);
        if( _picker->is_pick_init() ) _picker->set_name( XZ );
        draw_quad( _length );

        /* PLANE YZ*/
        if(_selected_axis == YZ )
            glColor4f(1.f, 1.f, 0.f, 0.5f);
        else
            glColor4f( 0.4f, 0.4f, 0.4f, 0.5f);

        glRotatef(90.f, 0.f, 1.f, 0.f);
        if( _picker->is_pick_init() ) _picker->set_name( YZ );
        draw_quad( _length );
    }
    glPopMatrix();
}

// -----------------------------------------------------------------------------

/// @return deem factor arrow (proportionnal to the angle between dir and axis)
static float alpha_arrow(const Vec3_cu& dir, const Vec3_cu& axis)
{
    float angle_x = std::abs( dir.dot( axis) );
    float alpha = 1.f - (std::max( angle_x, 0.99f) - 0.99f) * 100.f;
    return alpha;
}

// -----------------------------------------------------------------------------

void Gizmo_trans::draw(const Camera& cam)
{
    if(!_show) return;

    glPushMatrix();
    GLEnabledSave save_light(GL_LIGHTING  , true, false);
    GLEnabledSave save_depth(GL_DEPTH_TEST, true, false);
    GLEnabledSave save_blend(GL_BLEND     , true, true);
    GLBlendSave save_blend_eq;
    glAssert( glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA) );
    GLEnabledSave save_alpha(GL_ALPHA_TEST, true, true);
    GLEnabledSave save_textu(GL_TEXTURE_2D, true, false);
    GLLineWidthSave save_line();
    if( _picker->is_pick_init() )
        glLineWidth(10.f);
    else
        glLineWidth(1.f);

    const Vec3_cu org = _frame.get_translation();
    const float dist_cam =  !cam.is_ortho() ? (org-cam.get_pos()).norm() : cam.get_ortho_zoom() * _ortho_factor;

    glMultMatrixf(_frame.transpose().m);
    glScalef(dist_cam, dist_cam, dist_cam);

    float alpha_x = alpha_arrow( cam.get_dir(), _frame.x() );
    float alpha_y = alpha_arrow( cam.get_dir(), _frame.y() );
    float alpha_z = alpha_arrow( cam.get_dir(), _frame.z() );

    /* Axis X */
    if(_selected_axis == X )
        glColor3f(1.f, 1.f, 0.f);
    else
        glColor4f(1.f, 0.f, 0.f, alpha_x);
    glPushMatrix();
    glRotatef(90.f, 0.f, 1.f, 0.f);
    if( _picker->is_pick_init() ) _picker->set_name( X );
    draw_arrow(_radius, _length);
    glPopMatrix();

    /* Axis Y */
    if(_selected_axis == Y )
        glColor3f(1.f, 1.f, 0.f);
    else
        glColor4f(0.f, 1.f, 0.f, alpha_y);
    glPushMatrix();
    glRotatef(-90.f, 1.f, 0.f, 0.f);
    if( _picker->is_pick_init() ) _picker->set_name( Y );
    draw_arrow(_radius, _length);
    glPopMatrix();

    /* Axis Z */
    if(_selected_axis == Z )
        glColor3f(1.f, 1.f, 0.f);
    else
        glColor4f(0.f, 0.f, 1.f, alpha_z);
    glPushMatrix();
    if( _picker->is_pick_init() ) _picker->set_name( Z );
    draw_arrow(_radius, _length);
    glPopMatrix();

    draw_quads();

    glPopMatrix();
}

// -----------------------------------------------------------------------------

bool Gizmo_trans::select_constraint(const Camera& cam, int px, int py)
{
    if(!_show){
        _selected_axis = NOAXIS;
        return false;
    }

    slide_from(_frame, Vec2i_cu(px, py) );

    Transfo mvp = (cam.get_proj_transfo()*cam.get_eye_transfo()).transpose();
    _picker->begin(px, py, mvp.m);
        draw( cam );
    int id = _picker->end()._name;


    const Vec3_cu org = _frame.get_translation();
    Vec3_cu c = cam.project( org.to_point() );
    c.y = cam.height() - c.y;
    _pix_diff.x = (int)(c.x-px);
    _pix_diff.y = (int)(c.y-py);

    _selected_axis = NOAXIS;
    if( id > -1 )
    {
        _selected_axis = (Axis_t)id;
        switch(_selected_axis){
        case X : _axis = _frame.x(); break;
        case Y : _axis = _frame.y(); break;
        case Z : _axis = _frame.z(); break;
        case XY: _axis = _frame.z(); break;
        case XZ: _axis = _frame.y(); break;
        case YZ: _axis = _frame.x(); break;
        default: assert(false); break;
        }
        return true;
    }
    else
        return false;
}

// -----------------------------------------------------------------------------

TRS Gizmo_trans::slide(const Camera& cam, int px, int py)
{
    Vec2i_cu pix(px, py);

    if( _selected_axis != NOAXIS )
    {
        if( (pix - _start_pix).norm() < 0.0001f )
            return TRS();
        else if( !is_plane_constraint() )
            return slide_axis(_axis, cam, pix);
        else
            return slide_plane(_axis, cam, pix);
    }

    return TRS();
}

// -----------------------------------------------------------------------------

bool Gizmo_trans::is_plane_constraint()
{
    return _selected_axis == XY || _selected_axis == XZ || _selected_axis == YZ;
}

// -----------------------------------------------------------------------------

TRS Gizmo_trans::slide_axis(Vec3_cu axis_dir, const Camera& cam, Vec2i_cu pix)
{
    pix += _pix_diff;
    Ray_cu  ray = cam.cast_ray(pix.x, pix.y);
    Vec3_cu up  = cam.get_y();
    Vec3_cu dir = cam.get_dir();

    // Find a plane passing through axis_dir and as parrallel as possible to
    // the camera image plane
    Vec3_cu ortho = dir.cross(axis_dir);
    Vec3_cu p_normal;
    if(ortho.norm() < 0.00001) p_normal = axis_dir.cross( up  );
    else                       p_normal = axis_dir.cross(ortho);

    // Intersection between that plane and the ray cast by the mouse
    const Vec3_cu org = _start_frame.get_org();
    Inter::Plane axis_plane(p_normal, org);
    Inter::Line l_ray(ray._dir, ray._pos);
    Vec3_cu slide_point;
    bool inter = Inter::plane_line(axis_plane, l_ray, slide_point);

    // Project on axis_dir the intersection point
    dir = axis_dir.normalized();
    slide_point = org + dir * dir.dot(slide_point-org);

    if(inter )
        return TRS::translation( slide_point-org );
    else
        return TRS();
}

// -----------------------------------------------------------------------------

TRS Gizmo_trans::slide_plane(Vec3_cu normal  , const Camera& cam, Vec2i_cu pix)
{
    pix += _pix_diff;
    Ray_cu  ray = cam.cast_ray(pix.x, pix.y);

    // Intersection between that plane and the ray cast by the mouse
    const Vec3_cu org = _start_frame.get_org();
    Inter::Plane axis_plane(normal, org);
    Inter::Line l_ray(ray._dir, ray._pos);
    Vec3_cu slide_point;
    bool inter = Inter::plane_line(axis_plane, l_ray, slide_point);

    if(inter )
        return TRS::translation( slide_point-org );
    else
        return TRS();
}

// -----------------------------------------------------------------------------
