/*
 * This software is governed by the CeCILL-B license under French law and
 * abiding by the rules of distribution of free software.  You can  use, 
 * modify and/ or redistribute the software under the terms of the CeCILL-B
 * license as circulated by CEA, CNRS and INRIA at the following URL
 * "http://www.cecill.info" or the LICENCE.txt file present in this project.
*/

#ifndef GIZMO_TRANSLATION2_HPP__
#define GIZMO_TRANSLATION2_HPP__

#include "gizmo.hpp"
#include "port_glew.h"
#include "vec3_cu.hpp"
#include "camera.hpp"

/**
  @name Gizmo_trans
  @brief 3D frame to provide GUI for points translations.
  This class provide methods for translating objects in the GUI.
  It memories the current position and compute its new position
  given the mouse position and axis/plane constraint. It also gives means to
  draw a custom oriented frame which is represented by arrows. Each arrow can be
  selected.

  @see Gizmo
*/

class Gizmo_trans : public Gizmo {
public:

    enum Axis_t {X , Y , Z, XY, XZ, YZ, NOAXIS };

    Gizmo_trans( GlPick_FBO* picker );

    /// draw the current selected point and frame. Frame is represented with
    /// arrows (green, red and blue) made of a cylinder and a cone for the tip.
    /// Frame is drawn at the position specified by '_frame.get_translation()'.
    /// X, Y and Z arrows orientations are defined by 'set_frame()'.
    void draw(const Camera& cam);

    /// select the x or y or z frame axis given a camera and a mouse position
    /// updates attributes (_axis or _plane) and _constraint
    /// @return true if one of the axis is selected
    bool select_constraint(const Camera& cam, int px, int py);

    /// reset the selected constraint set by select_constraint()
    void reset_constraint(){ _selected_axis  = NOAXIS; }

    /// @brief slide point given the current constraint (plane or axis)
    /// If constraint has been selected this will move the selected point to
    /// its new position by following the constraint and keeping the frame
    /// as close as possible under the mouse position (px, py)
    /// @return the translation made by the frame in world coordinates
    /// (when clicked on)
    /// @see select_constraint()
    TRS slide(const Camera& cam, int px, int py);

private:
    TRS slide_axis (Vec3_cu axis_dir, const Camera& cam, Vec2i_cu pix);
    TRS slide_plane(Vec3_cu normal  , const Camera& cam, Vec2i_cu pix);

    void draw_quads();

    bool is_plane_constraint();

    void init_attr();

    Vec3_cu _color;       ///< point color
    float    _size;       ///< point size when drawing

    /// Current frame in which the point is moved
    //@{
    float _length;         ///< length of the arrows representing the frame
    float _radius;         ///< radius of the arrows representing the frame
    float _ortho_factor;
    //@}

    Axis_t       _selected_axis;  ///< current selected axis

    Vec2i_cu     _pix_diff;       ///< pixels to substract when sliding
    Vec3_cu      _axis;           ///< current axis direction for movements
    Vec3_cu      _plane;          ///< current plane normal for movements
};

#endif // GIZMO_TRANSLATION2_HPP__
