/*
 * This software is governed by the CeCILL-B license under French law and
 * abiding by the rules of distribution of free software.  You can  use, 
 * modify and/ or redistribute the software under the terms of the CeCILL-B
 * license as circulated by CEA, CNRS and INRIA at the following URL
 * "http://www.cecill.info" or the LICENCE.txt file present in this project.
*/

#ifndef MSGE_STACK_HPP_
#define MSGE_STACK_HPP_

#include <deque>
#include <QString>
#include <QTimer>
#include <QTime>
#include <QFont>
#include <QGLWidget>

#include "text_renderer.hpp"

/** @class Msge_stack
    @brief Utility to print messages a limited amount of time on a viewport

    This class is an utility to prompt messages over the viewport.
    Messages are stacked and poped at regular intervals
*/

#define MSGE_LIFE_TIME 3000 // in ms
#define FONT_SIZE 17

class Msge_stack : public QObject{
    Q_OBJECT
public:

    Msge_stack(const QGLWidget *gl_widget) : QObject() {
        QFont font;
        font.setStyleHint( QFont::Courier, QFont::PreferAntialias );
        font.setPixelSize(FONT_SIZE);
        _text_renderer.setup( gl_widget, font );
        QObject::connect(&_timer, SIGNAL(timeout()), this, SLOT(time_out()));
    }

    /// Draw the message stack at (x,y) position. (bottom left is (0,0))
    void draw(int x, int y){
        _text_renderer.begin();
        for (unsigned i = 0; i < _msge.size(); ++i)
            _text_renderer.print( x, y-i*(FONT_SIZE+1), _msge[i] );
        _text_renderer.end();
    }

    /// Push a message 'msge' to the stack it will be poped after a delay equal
    /// to MSGE_LIFE_TIME.
    /// @param add_date If true concatenate the current time before the string
    /// 'msge'
    void push(const QString& msge, bool add_date = false){
        _timer.start(MSGE_LIFE_TIME);
        QString date = "";
        if(add_date)
            date = "["+QTime::currentTime().toString("hh:mm:ss")+"] ";
        _msge.push_back(date+msge);
        emit updateGL();
    }

    /// Erase all currently displayed messages
    void clear()
    {
        _msge.clear();
        time_out();
    }

signals:
    /// Emited when messages changes and needs to be drawn again
    void updateGL();

private slots:
    void time_out(){
        if(_msge.size() == 0) _timer.stop();
        else                  _msge.pop_front();

        emit updateGL();
    }
private:
    /// Render text on opengl window
    TextRenderer _text_renderer;
    /// Stack of messages progressively erased
    std::deque<QString> _msge;
    /// The timer pops messages at constant intervalles
    QTimer _timer;
};

#endif // MSGE_STACK_HPP_
