/*
 * This software is governed by the CeCILL-B license under French law and
 * abiding by the rules of distribution of free software.  You can  use, 
 * modify and/ or redistribute the software under the terms of the CeCILL-B
 * license as circulated by CEA, CNRS and INRIA at the following URL
 * "http://www.cecill.info" or the LICENCE.txt file present in this project.
*/

#include "main_window.hpp"

#include <QFileDialog>
#include <QColorDialog>
#include <QMessageBox>
#include <QString>

#include "common/tools/popup_ok_cancel.hpp"
#include "vec3_cu.hpp"
#include "camera.hpp"
#include "OGL_viewports.hpp"
#include "g_scene_tree.hpp"
#include "obj_hrbf.hpp"

extern std::string g_icons_dir;

void load_test_mesh(); // DEBUG to be deleted

// -----------------------------------------------------------------------------

Main_window::Main_window(QWidget *parent) :
    QMainWindow(parent)
{
    //--------------
    setupUi(this);
    //--------------
    setup_attrs();
    setup_viewports();
    setup_toolbar();
    setup_main_window();
    //--------------

    // Desactivate GUI parts
    enable_gui_animesh( false );
    enable_gui_mesh   ( false );

    // DEBUG =======================================
    //load_fbx("./resource/meshes/dana/dana_from_blender.fbx");

    //load_mesh("./resource/meshes/egg.obj");
    //load_mesh("./resource/meshes/egg.obj");
    //load_mesh("./resource/meshes/egg.obj");

    g_scene_tree->register_obj( new Obj_HRBF() );
    /*
    for (int i = 0; i < 6; ++i) {
        load_mesh("./resource/meshes/cylindre_textured/cylindre_chessboard.obj");
    }
    */
    // DEBUG =======================================
}

// -----------------------------------------------------------------------------

void Main_window::setup_attrs()
{
    _selection_comboBox = 0;
    _viewports          = 0;

    _is_ctrl_pushed = false;
    _is_alt_pushed = false;
    _is_tab_pushed = false;
    _is_maj_pushed = false;
    _is_space_pushed = false;
}

// -----------------------------------------------------------------------------

void Main_window::setup_toolbar()
{
    // Connect toolButtons for selection
#if 0
    Widget_selection* select = toolBar->_wgt_select;
    QObject::connect(select->toolB_select_point, SIGNAL(pressed()), this, SLOT(selection_toolb_point()));
    QObject::connect(select->toolB_select_circle, SIGNAL(pressed()), this, SLOT(selection_toolb_circle()));
    QObject::connect(select->toolB_select_square, SIGNAL(pressed()), this, SLOT(selection_toolb_square()));
#endif

    Widget_viewports* view = toolBar->_wgt_viewport;
    QObject::connect(view->toolB_single, SIGNAL(pressed()), this, SLOT(viewport_toolb_single()));
    QObject::connect(view->toolB_doubleV, SIGNAL(pressed()), this, SLOT(viewport_toolb_doubleV()));
    QObject::connect(view->toolB_doubleH, SIGNAL(pressed()), this, SLOT(viewport_toolb_doubleH()));
    QObject::connect(view->toolB_four, SIGNAL(pressed()), this, SLOT(viewport_toolb_four()));

    // Connect toolButtons for gizmo
    Widget_gizmo* gizmo = toolBar->_wgt_gizmo;
    QObject::connect(gizmo->toolB_show_gizmo, SIGNAL(toggled(bool)), this, SLOT(show_all_gizmo(bool)));
    QObject::connect(gizmo->toolB_translate, SIGNAL(pressed()), this, SLOT(set_gizmo_trans()));
    QObject::connect(gizmo->toolB_rotate, SIGNAL(pressed()), this, SLOT(set_gizmo_rot()));
    QObject::connect(gizmo->toolB_trackball, SIGNAL(pressed()), this, SLOT(set_gizmo_trackball()));
    QObject::connect(gizmo->toolB_scale, SIGNAL(pressed()), this, SLOT(set_gizmo_scale()));

    // Connect toolButtons for the rendering mode
#if 0
    Widget_render_mode* render = toolBar->_wgt_rd_mode;
    QObject::connect(render->toolB_wire_transc, SIGNAL(pressed()), this, SLOT(rd_mode_toolb_wire_transc()));
    QObject::connect(render->toolB_wire, SIGNAL(pressed()), this, SLOT(rd_mode_toolb_wire()));
    QObject::connect(render->toolB_solid, SIGNAL(pressed()), this, SLOT(rd_mode_toolb_solid()));
    QObject::connect(render->toolB_tex, SIGNAL(pressed()), this, SLOT(rd_mode_toolb_tex()));

    // Fitting buton
    QObject::connect(toolBar->_wgt_fit->toolB_fitting, SIGNAL(toggled(bool)), this, SLOT(toggle_fitting(bool)));

    // Connect combo box for the pivot mode / gizmo mode
    QObject::connect(toolBar->_pivot_comboBox, SIGNAL(currentIndexChanged(int)), this, SLOT(pivot_comboBox_currentIndexChanged(int)));
    QObject::connect(toolBar->_pivot_gizmo_comboBox, SIGNAL(currentIndexChanged(int)), this, SLOT(gizmo_pivot_comboBox_currentIndexChanged(int)));
#endif
    QObject::connect(toolBar->_dir_gizmo_comboBox, SIGNAL(currentIndexChanged(int)), this, SLOT(gizmo_dir_comboBox_currentIndexChanged(int)));
}

// -----------------------------------------------------------------------------

void Main_window::setup_viewports()
{
    _viewports = new OGL_viewports(viewports_frame, this);
    viewports_frame->layout()->addWidget(_viewports);
}

// -----------------------------------------------------------------------------

void Main_window::setup_main_window()
{
    setWindowTitle("Hermite Radial Basis Function (HRBF) Toy Application");
    resize(1300, 800);
    QIcon icon(QString(g_icons_dir.c_str())+"/logo/red_logo_white.svg");
    QWidget::setWindowIcon( icon );
}

// -----------------------------------------------------------------------------

void Main_window::update_viewports()
{
    _viewports->updateGL();
}

// -----------------------------------------------------------------------------

void Main_window::paint_toggled(bool state)
{
    if(state){
        toolBar->_wgt_select->toolB_select_circle->setChecked(true);
        // trigger corresponding slot
        selection_toolb_circle();
        toolBar->_wgt_select->setEnabled( false );
    }
    else
    {
        toolBar->_wgt_select->setEnabled( true );
    }
}

// MANUAL SLOTS ################################################################

void Main_window::selection_toolb_point()
{
    Vec_viewports& list = _viewports->get_viewports();

    for(unsigned i = 0; i < list.size(); i++)
    {
        list[i]->setMouseTracking(false);
        list[i]->set_selection(EOGL_widget::MOUSE);
    }
    update_viewports();
}

void Main_window::selection_toolb_circle()
{
    Vec_viewports& list = _viewports->get_viewports();

    for(unsigned i = 0; i < list.size(); i++)
    {
        list[i]->setMouseTracking(false);

        list[i]->set_selection(EOGL_widget::CIRCLE);
        list[i]->setMouseTracking(true);
    }
    update_viewports();
}

void Main_window::selection_toolb_square(){

}

void Main_window::viewport_toolb_single(){
    _viewports->set_viewports_layout(OGL_viewports::SINGLE);
    _viewports->updateGL();
}

void Main_window::viewport_toolb_doubleV(){
    _viewports->set_viewports_layout(OGL_viewports::VDOUBLE);
    _viewports->updateGL();
}

void Main_window::viewport_toolb_doubleH(){
    _viewports->set_viewports_layout(OGL_viewports::HDOUBLE);
    _viewports->updateGL();
}

void Main_window::viewport_toolb_four(){
    _viewports->set_viewports_layout(OGL_viewports::FOUR);
    _viewports->updateGL();
}

void Main_window::rd_mode_toolb_tex()
{
    //OGL_widget* wgl = _viewports->active_viewport();

    update_viewports();
}

void Main_window::rd_mode_toolb_solid()
{
    //OGL_widget* wgl = _viewports->active_viewport();

    update_viewports();
}

void Main_window::rd_mode_toolb_wire()
{
    //OGL_widget* wgl = _viewports->active_viewport();

    update_viewports();
}

void Main_window::rd_mode_toolb_wire_transc()
{
    //OGL_widget* wgl = _viewports->active_viewport();

    update_viewports();
}

void Main_window::show_all_gizmo(bool checked)
{
    _viewports->show_gizmo(checked);
    update_viewports();
}

void Main_window::set_gizmo_trans(){
    _viewports->set_gizmo(Gizmo::TRANSLATION);
}

void Main_window::set_gizmo_rot(){
    _viewports->set_gizmo(Gizmo::ROTATION);
}

void Main_window::set_gizmo_trackball(){
    _viewports->set_gizmo(Gizmo::TRACKBALL);
}

void Main_window::set_gizmo_scale(){
    _viewports->set_gizmo(Gizmo::SCALE);
}

void Main_window::toggle_fitting(bool /*checked*/){

    update_viewports();
}

void Main_window::pivot_comboBox_currentIndexChanged(int idx)
{
    int val = toolBar->_pivot_comboBox->itemData( idx ).toInt();
    _viewports->set_cam_pivot((EOGL_widget::Pivot_t)val);
}

void Main_window::gizmo_pivot_comboBox_currentIndexChanged(int idx)
{
    int val = toolBar->_pivot_gizmo_comboBox->itemData( idx ).toInt();
    _viewports->set_gizmo_pivot((EIO_Selection::Pivot_t)val);
}

void Main_window::gizmo_dir_comboBox_currentIndexChanged(int idx)
{
    int val = toolBar->_dir_gizmo_comboBox->itemData( idx ).toInt();
    _viewports->set_gizmo_dir((EIO_Selection::Dir_t)val);
}

void Main_window::active_viewport(int id)
{
    static int id_prev = -1;
    // Update necessary only if the active viewport is changed
    if(id_prev == id) return;


    id_prev = id;
}

// AUTOMATIC SLOTS #############################################################

void Main_window::on_actionExit_triggered()
{
    exit(0);
}

#include "marching_cubes.hpp"
void Main_window::on_spinBox_valueChanged(int arg1)
{
    Marching_cubes::set_resolution( Vec3i_cu(arg1) );
    _viewports->updateGL();
}

void Main_window::on_spinB_aperture_valueChanged(int arg1)
{
    _viewports->active_viewport()->camera()->set_fov_deg( arg1 );
    _viewports->updateGL();
}
