/*
 * This software is governed by the CeCILL-B license under French law and
 * abiding by the rules of distribution of free software.  You can  use, 
 * modify and/ or redistribute the software under the terms of the CeCILL-B
 * license as circulated by CEA, CNRS and INRIA at the following URL
 * "http://www.cecill.info" or the LICENCE.txt file present in this project.
*/

#include "main_window.hpp"

/*
    Implements what's concerning the file menu of the MainWindow class
*/

#include <QFileDialog>
#include <QColorDialog>
#include <QMessageBox>

#include <cassert>
#include <algorithm>

#include "common/tools/popup_ok_cancel.hpp"
#include "vec3_cu.hpp"
#include "OGL_viewports.hpp"
#include "loader.hpp"
#include "generic_loader.hpp"
#include "obj_loader.hpp"
#include "weights_loader.hpp"
#include "globals.hpp"
#include "std_utils.hpp"
#include "g_scene_tree.hpp"



// -----------------------------------------------------------------------------

static bool supported(const Loader::Generic_file& file, QWidget* parent)
{
    bool s = file.supported();
    if(!s){
        std::string str = Std_utils::file_ext( file.file_path() );
        QString ext( str.c_str() );
        QMessageBox::warning(parent, "Error", "Unsupported file type: '"+ext+"'");
    }
    return s;
}

// FILE SLOT ===================================================================

void Main_window::load_mesh(const QString& fileName)
{
    if( fileName.size() != 0)
    {
        if( !QFile(fileName).exists() ){
            QMessageBox::warning(this, "Error", "Can't find file: '"+fileName+"'");
            return;
        }

        Loader::Generic_file file( fileName.toStdString() );
        if( !supported(file, this) ) return;

        EObj::Flags f = file.fill_scene( *g_scene_tree, EObj::MESH );

        if( EObj::test(f, EObj::MESH) )
        {
            enable_gui_animesh( false );
            enable_gui_mesh( true );
        }else
            QMessageBox::information(this, "Error", "It doesn't seem to be any mesh in this file.");
    }
    update_viewports();
}

// -----------------------------------------------------------------------------

void Main_window::load_fbx(const QString& fileName)
{
    if( fileName.size() != 0)
    {
        QString ext = QFileInfo(fileName).suffix().toLower();
        if( ext != "fbx" ){
            QMessageBox::information(this, "Error !", "this is not a FBX file: "+fileName+"\n");
            return;
        }

        if( QFile::exists(fileName) )
        {
            Loader::Generic_file file( fileName.toStdString() );
            EObj::Flags t = file.fill_scene( *g_scene_tree );

            if( EObj::test(t, EObj::MESH)     ) enable_gui_mesh   ( true );
            if( EObj::test(t, EObj::SKELETON) ) enable_gui_animesh( true );

            // Load animations
//            std::vector<Loader::Base_anim_eval*> anims;
//            loader.get_animations( anims );
//            toolBar_frame->set_anim_list( anims );
        }
        else
        {
            QMessageBox::information(this, "Error !", "Can't' find "+fileName+"\n");
            return;
        }
    }
}

// -----------------------------------------------------------------------------

void Main_window::on_actionLoad_Obj_triggered()
{
    QString fileName = QFileDialog::getOpenFileName(this,
                                                    tr("Load mesh"),
                                                    "./resource/meshes",
                                                    tr("*.off *.obj") );
    if( fileName.size() != 0)
    {
        load_mesh( fileName );
    }

    update_viewports();
}
