/*
 * This software is governed by the CeCILL-B license under French law and
 * abiding by the rules of distribution of free software.  You can  use, 
 * modify and/ or redistribute the software under the terms of the CeCILL-B
 * license as circulated by CEA, CNRS and INRIA at the following URL
 * "http://www.cecill.info" or the LICENCE.txt file present in this project.
*/

#ifndef __CAMERA_H__
#define __CAMERA_H__

#include "ray_cu.hpp"
#include "vec3_cu.hpp"
#include "transfo.hpp"

/** @class Camera
  @brief Handles camera movements, and various related parameters
*/

class Camera {
public:

    Camera();

    // =========================================================================
    /// @name Moving the camera
    // =========================================================================

    /// update the direction from a rotation
    /// @param yaw_    rotation around y
    /// @param pitch_  rotation around x
    void update_dir(float yaw, float pitch, float roll);

    void set_dir(const Vec3_cu& dir);

    /// update the position from a direction in local space
    void update_pos(Vec3_cu mod_local_);

    /// Sets origin of the camera
    void set_pos(const Vec3_cu& org);

    /// dir and up must be orthogonal to each other
    void set_dir_and_up(const Vec3_cu& dir, const Vec3_cu& up);

    /// This sets the camera frame as to look at the point 'aimed'
    void lookat(const Vec3_cu& aimed);

    void roll(float theta);

    void local_strafe(float x, float y, float z);

    Vec3_cu get_pos() const;

    Vec3_cu get_dir() const;

    Vec3_cu get_y() const;

    Vec3_cu get_x() const;

    Transfo get_frame() const;

    /// Apply a global transformation to the camera frame
    void transform(const Transfo& gtransfo);

    // =========================================================================
    /// @name Common camera operations
    // =========================================================================

    /// This call gluLookAt
    void lookat() const;

    /// Multiply the current OpenGL matrix with the ortho projection matrix
    /// @see gl_perspective_mult() gl_mult_projection()
    void gl_ortho_mult() const;

    /// Multiply the current OpenGL matrix with the perspective projection matrix
    /// @see gl_ortho_mult() gl_mult_projection()
    void gl_perspective_mult() const;

    /// Multiply the current OpenGL matrix with the projection matrix of the
    /// camera (either perspective or ortho acording to is_ortho() state)
    void gl_mult_projection() const;

    /// Compute the projection matrix. (as in OpenGL from eye coordinates
    /// to normalized device coordinates)
    Transfo get_proj_transfo() const;

    /// Get the view matrix (As computed with gluLookAt from model coordinates
    /// to eye coordinates
    Transfo get_eye_transfo() const;

    /// Get the viewport matrix (as in OpenGL from normalized device coordinates
    /// to window coordinates)
    /// @note z is mapped between [0 1] as in the default value of openGL
    /// for glDepthRange()
    Transfo get_viewport_transfo() const;

    /// Project 'p' in world coordinates to the camera image plane
    /// (in window coordinates)
    /// @return p' = viewport_transfo * proj_transfo * eye_transfo * p
    Point_cu project(const Point_cu& p) const;

    /// unProject 'p' in window coordinates to world coordinates
    /// @return p' = (viewport_transfo * proj_transfo * eye_transfo)^-1 * p
    Point_cu un_project(const Point_cu& p) const;

    // TODO: (py) seems to be inverted (its in qt coords system 0,0 at upper left)
    // -> would more consistent to be in opengl coords 0,0 at bottom left
    /// Cast a ray from camera : px, py are the pixel position you want
    /// to cast a ray from
    Ray_cu cast_ray(int px, int py) const;

    // =========================================================================
    /// @name Frustum characteristics
    // =========================================================================

    /// Set the camera aperture in degrees
    void set_fov_deg(float f);

    /// Set the camera aperture in radians
    void set_fov_rad(float f);


    int width()  const { return _width;  }
    int height() const { return _height; }

    float get_near() const { return _near;  }
    float get_far()  const { return _far;   }

    /// Aperture along y axis;
    float fovy() const { return _fov; }

    /// Aperture along x axis;
    float fovx() const {  return _fov * ((float)_width / (float)_height);  }

    bool is_ortho() const { return _ortho_proj; }

    void set_ortho(bool s) { _ortho_proj = s; }

    void set_near(float n){ _near = n; }

    void set_far(float f){ _far = f; }

    void set_viewport(int x, int y, int w, int h){
        _width  = w; _height = h; _offy   = y; _offx   = x;
    }

    /// Frustum width in orthogonal projection @see is_ortho()
    /// @{
    float get_ortho_zoom() const;
    void  set_ortho_zoom(float width);
    /// @}

    void print() const;

    // -------------------------------------------------------------------------
    /// @name Attributes
    // -------------------------------------------------------------------------
private:
    Vec3_cu _pos;    ///< position
    Vec3_cu _dir;    ///< sight direction (z axis)
    Vec3_cu _x, _y;  ///< other vectors to get the frame

    float _fov;           ///< openGl opening angle along Y axis (in radian)
    float _near;          ///< near plane distance
    float _far;           ///< far plane distance
    float _ortho_zoom;    ///< frustum width zoom in orthogonal projection

    bool _ortho_proj;     ///< does orthogonal projection enabled ?

    int _width;  ///< pixel width of the camera viewport
    int _height; ///< pixel height of the camera viewport

    int _offx;   ///< x offset of the camera viewport
    int _offy;   ///< y offset of the camera viewport
};

#endif
