/*
 * This software is governed by the CeCILL-B license under French law and
 * abiding by the rules of distribution of free software.  You can  use, 
 * modify and/ or redistribute the software under the terms of the CeCILL-B
 * license as circulated by CEA, CNRS and INRIA at the following URL
 * "http://www.cecill.info" or the LICENCE.txt file present in this project.
*/

#include "scene_renderer.hpp"

#include "gl_utils.hpp"
#include "color.hpp"
#include "marching_cubes.hpp"
#include "obj_skeleton.hpp"

// -----------------------------------------------------------------------------

Scene_renderer::Scene_renderer(const Camera* cam,
                               Scene_tree* tree,
                               int width,
                               int height ) :
    _cam( cam ),
    _tree( tree ),
    _alpha_strength(0.5f)
{
    // initialize context
    _ctx = new Render_context(width, height);
    // New picker
    _picker = new Scene_picker(this);
    // Shaders/texs/VbOs intialization for marching cubes rendering on shaders
    Marching_cubes::init();
}

// -----------------------------------------------------------------------------

Scene_renderer::~Scene_renderer()
{
    delete _ctx;
    delete _picker;
    _ctx = 0;
    _picker = 0;

    Marching_cubes::clean();
}

// -----------------------------------------------------------------------------

void Scene_renderer::draw_object(Obj* o, const Transfo& eye) const
{
    // transform
    const Transfo tr = _global_tr.find(o)->second;
    glAssert( glLoadMatrixf( (eye * tr).transpose().m ) );

    o->draw();
}

// -----------------------------------------------------------------------------

void Scene_renderer::draw_object_outline(Obj* o) const
{
    // Prepare to write in stencil
    GLEnabledSave lighting(GL_LIGHTING, true, false);
    GLEnabledSave depth(GL_DEPTH_TEST, true, false);
    GLDepthMaskSave depth_mask(GL_FALSE);
    GLEnabledSave stencil(GL_STENCIL_TEST, true, true);
    GLColorMaskSave mask(GL_FALSE, GL_FALSE, GL_FALSE, GL_FALSE);
    glStencilOp(GL_REPLACE, GL_REPLACE, GL_REPLACE);
    glStencilFunc(GL_ALWAYS, 1, 0xffffffff);
    // write stencil with wires to 1
    draw_object_wires( o );
    glStencilFunc(GL_ALWAYS, 0, 0xffffffff);
    // write stencil with plain object to 0
    GLPolygonModeSave poly_mode(GL_FILL);
    o->draw();

    // Draw outline
    glEnable(GL_DEPTH_TEST);
    glColorMask( GL_TRUE, GL_TRUE, GL_TRUE, GL_TRUE );
    glDepthMask( GL_TRUE );
    glStencilFunc(GL_EQUAL, 1, 0xffffffff); // draw if == 1
    glStencilOp(GL_KEEP, GL_KEEP, GL_KEEP);
    draw_object_wires( o );
}

// -----------------------------------------------------------------------------

void Scene_renderer::draw_outlined_objects( const Transfo& eye) const
{
    for(unsigned i = 0; i < _selected_objs.size(); ++i)
    {
        Obj* o = _selected_objs[i];
        const Transfo tr = _global_tr.find(o)->second;
        glAssert( glLoadMatrixf( (eye * tr).transpose().m ) );

        if( o->type_object() == EObj::IMPLICIT )
        {
            Node_implicit_surface* node = dynamic_cast<Node_implicit_surface*>( o );
            glColor3f(1.f, 0.66f, 0.251f);
            GLLineWidthSave line_w(1.5f);
            Gl_utils::draw( node->get_bbox() );
        }
        else
            draw_object_outline( o );
    }
}

// -----------------------------------------------------------------------------

void Scene_renderer::draw_object_wires(Obj* o) const
{

    if(o == _picker->get_active())
        Color::red().set_gl_state();
    else
        (Color::red()*0.5).set_gl_state();

    GLLineWidthSave line_width(3.0f);
    GLPolygonModeSave poly_mode(GL_LINE);

    Obj::Use_shader save(false);
    Obj::Save_state save_state(o, Obj::RENDER);
    o->draw();
}

// -----------------------------------------------------------------------------

void Scene_renderer::draw_wired_objects(const Transfo& eye) const
{
    for(unsigned i = 0; i < _wires_objs.size(); ++i) {
        GLLineWidthSave line_width(1.0f);
        GLPolygonModeSave poly_mode(GL_LINE);
        GLEnabledSave lighting(GL_LIGHTING, true, false);
        draw_object(_wires_objs[i], eye);
    }
}

// -----------------------------------------------------------------------------

void Scene_renderer::draw_parent_dot_lines(const Transfo& eye) const
{
    glLineStipple(5, 0xAAAA);
    GLEnabledSave save_line(GL_LINE_STIPPLE, true, true);
    GLLineWidthSave line_width(2.0f);
    GLEnabledSave lighting(GL_LIGHTING, true, false);
    GLEnabledSave texture(GL_TEXTURE_2D, true, false);

    glAssert( glLoadMatrixf( Transfo::identity().transpose().m ) );
    Scene_tree::iterator it = _tree->begin();
    for(; it != _tree->end(); ++it)
    {
        const Obj* obj = *it;
        const Obj* prt = obj->parent();

        // Don't draw a line between root and its sons
        if(prt->parent() == 0) continue;

        Vec3_cu p0 = _global_tr.find(obj)->second.get_org();
        Vec3_cu p1 = _global_tr.find(prt)->second.get_org();

        p0 = (Vec3_cu)(eye * p0.to_point());
        p1 = (Vec3_cu)(eye * p1.to_point());

        glBegin(GL_LINES);
        glColor3f(1.f, 0.f, 0.f);
        glVertex3f(p0.x, p0.y, p0.z);
        glVertex3f(p1.x, p1.y, p1.z);
        glEnd();
    }
}

// -----------------------------------------------------------------------------

void Scene_renderer::draw_polygonize_obj(Obj* o, const Transfo& eye) const
{
    const Transfo tr = _global_tr.find(o)->second;
    glAssert( glLoadMatrixf( (eye * tr).transpose().m ) );

    Node_implicit_surface* node = dynamic_cast<Node_implicit_surface*>( o );


    Marching_cubes::fill_3D_grid_with_scalar_field( node );
    Marching_cubes::render_scalar_field();

    //Marching_cubes::render_scalar_field_cpu( node );
    if( !o->_selected ) {
        Color::white().set_gl_state();
        GLLineWidthSave line_w(1.0f);
        Gl_utils::draw( node->get_bbox() );
    }
}

// -----------------------------------------------------------------------------

bool Scene_renderer::draw()
{

    // Must be done first:
    // {
    update_scene( _tree );
    // }

    const int width  = _ctx->width();
    const int height = _ctx->height();
    const Transfo eye = _cam->get_eye_transfo();
    clear_background();
    setup_camera();

    if( _tree == 0) return false;

    ///////////////////
    // depth peeling //
    if( _peel_objs.size() > 0)
    {
        Peel_functor peel_fun(this, _ctx, _peel_objs);
        _ctx->peeler()->set_render_func( &peel_fun );
        // reset background
        _ctx->peeler()->set_background(width, height,
                                       _ctx->pbo_color(),
                                       _ctx->pbo_depth());
        _ctx->peeler()->peel( _alpha_strength );
    }

    // Copy depth of raytracing
    /*
    if( _raytrace_objs.size() > 0 )
    {
        glAssert( glDisable(GL_DEPTH_TEST) );
        GLColorMaskSave color_mask(GL_FALSE, GL_FALSE, GL_FALSE, GL_FALSE);
        _ctx->pbo_depth()->bind();
        glAssert( glDrawPixels(width, height,GL_DEPTH_COMPONENT,GL_FLOAT,0) );
        _ctx->pbo_depth()->unbind();
    }
    */

    glAssert( glEnable(GL_DEPTH_TEST) );

    // Draw depth of peeled objects
    if( _peel_objs.size() > 0 )
    {
        GLColorMaskSave color_mask(GL_FALSE, GL_FALSE, GL_FALSE, GL_FALSE);
        GLPolygonOffsetSave offset(1.1f, 4.f);
        GLEnabledSave enable_offset(GL_POLYGON_OFFSET_FILL, true, true);
        Peel_functor peel_fun(this, _ctx, _peel_objs);
        peel_fun.draw_transc_objs();
    }

    // Draw implicit surfaces with marching cube
    for(unsigned i = 0; i < _polygonize_objs.size(); ++i)
        draw_polygonize_obj(_polygonize_objs[i], eye);

    // Outlines of selected objects
    draw_outlined_objects(eye);

    // Objects wireframe
    draw_wired_objects(eye);

    draw_parent_dot_lines(eye);

    ////////////
    // Raster //
    for(unsigned i = 0; i < _raster_objs.size(); ++i)
        draw_object(_raster_objs[i], eye);

    //if(_display._ssao && !refresh) redraw_with_ssao();

    // We should return with only the camera matrix setup
    glAssert( glLoadMatrixf( eye.transpose().m ));

    return false;
}

// -----------------------------------------------------------------------------

void Scene_renderer::update_scene(const Scene_tree* tree)
{
    _global_tr.clear();
    _raster_objs.clear();
    _wires_objs.clear();
    _peel_objs.clear();
    _selected_objs.clear();
    _polygonize_objs.clear();

    if(tree == 0) return;
    Scene_tree::compute_global_tr(tree->root(), _global_tr);

    Scene_tree::const_iterator it = tree->begin();
    for( ; it != tree->end(); ++it)
    {
        Obj* object = *it;

        // Object drawing is disabled :
        if( !object->state(Obj::RENDER) ) continue;

        switch( object->type_render() )
        {
        case (Obj::RASTER):
        {
            _raster_objs.push_back(object);
            if( object->state( Obj::WIRES ) )
                _wires_objs.push_back(object);
        } break;

        case (Obj::PEEL):       { _peel_objs.push_back(object);       } break;
        case (Obj::POLYGONIZE): {
            _polygonize_objs.push_back(object);
            // Some things aside from the scalar field could be drawn using
            // the draw() method
            _raster_objs    .push_back(object);
        } break;
        default: assert(false); break;
        }

        if( object->_selected)
            _selected_objs.push_back( object );

        if( object->state(Obj::WIRES) )
            _wires_objs.push_back( object );
    }
}

// -----------------------------------------------------------------------------

void Scene_renderer::clear_background()
{
    Color cl( 0.1f );
    glAssert( glClearColor(cl.r, cl.g, cl.b, cl.a) );
    glAssert( glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT | GL_STENCIL_BUFFER_BIT) );
}

// -----------------------------------------------------------------------------

void Scene_renderer::setup_camera()
{
    glAssert( glMatrixMode(GL_PROJECTION) );
    glAssert( glLoadIdentity() );
    glViewport(0, 0, _cam->width(), _cam->height());
    _cam->gl_mult_projection();

    glAssert( glMatrixMode(GL_MODELVIEW) );
    glLoadIdentity();

    // TODO: handle lighting the opengl 3.2 way
    float _light0_ambient [4] = { 0.1f, 0.1f, 0.1f, 1.0f };//{ 0.2f, 0.2f, 0.2f, 1.0f };
    float _light0_diffuse [4] = { 1.0f, 1.0f, 1.0f, 1.0f };
    float _light0_specular[4] = { 1.0f, 1.0f, 1.0f, 1.0f };
    float _light0_position[4] = { 0.0f, 0.0f, 0.0f, 1.0f };

    glEnable(GL_LIGHTING);
    glEnable(GL_LIGHT0);
    glLightfv(GL_LIGHT0, GL_AMBIENT , _light0_ambient );
    glLightfv(GL_LIGHT0, GL_DIFFUSE , _light0_diffuse );
    glLightfv(GL_LIGHT0, GL_SPECULAR, _light0_specular);
    glLightfv(GL_LIGHT0, GL_POSITION, _light0_position);
    GL_CHECK_ERRORS();

    _cam->lookat();
}
