/*
 * This software is governed by the CeCILL-B license under French law and
 * abiding by the rules of distribution of free software.  You can  use, 
 * modify and/ or redistribute the software under the terms of the CeCILL-B
 * license as circulated by CEA, CNRS and INRIA at the following URL
 * "http://www.cecill.info" or the LICENCE.txt file present in this project.
*/

#ifndef SCENE_RENDER_HPP__
#define SCENE_RENDER_HPP__

#include "render_context.hpp"
#include "scene_tree.hpp"
#include "camera.hpp"
#include "glsave.hpp"
#include "glpick.hpp"

/**
 * @brief Opengl rendering of a Scene_tree
 *
 * @see Scene_tree Obj
 */
class Scene_renderer {
public:


    Scene_renderer(const Camera* cam, Scene_tree* tree, int width, int height );
    ~Scene_renderer();

    /// @return if you need to call again draw() to complete the rendering for
    /// the current frame
    bool draw();

    // -------------------------------------------------------------------------
    /// @name Getters & Setters
    // -------------------------------------------------------------------------

    Picker* get_picker(){ return _picker; }
    Render_context* render_context() const { return _ctx; }

    void set_scene_tree(Scene_tree* tree){ _tree = tree; }
    Scene_tree* get_scene_tree() const { return _tree; }

    void set_alpha_strength(float a) {
        _alpha_strength = std::max(std::min(a, 1.f), 0.f);
    }

private:

    // -------------------------------------------------------------------------
    /// @name Tools
    // -------------------------------------------------------------------------

    /// Given a scene graph update the scene to be rendered
    void update_scene(const Scene_tree* tree);

    void clear_background();

    void setup_camera();

    /// Draw an object given its states/materials and positions
    /// @param o : the object to draw
    /// @param eye : camera matrix to go from world to view coordinates
    void draw_object(Obj* o, const Transfo& eye) const;

    /// Draws the object wires in the outline color
    void draw_object_wires(Obj* o) const;

    /// Draw the object's outline
    void draw_object_outline(Obj* o) const;

    /// draw every selected objects outline
    /// @param eye : camera matrix to go from world to view coordinates
    void draw_outlined_objects(const Transfo& eye) const;

    /// Draw every object set with wire rendering
    void draw_wired_objects(const Transfo& eye) const;

    /// Draw dot lines between an object and its parent
    void draw_parent_dot_lines(const Transfo& eye) const;

    /// Draw implicit surfaces set to be rendered by polygonisation
    void draw_polygonize_obj(Obj* obj, const Transfo& eye) const;

    // -------------------------------------------------------------------------
    /// @name Inner class
    // -------------------------------------------------------------------------

    /**
     * @name Scene_picker
     * @brief specialization for picking an object into the scene
     * @see Picker
     */
    class Scene_picker : public Picker {
    public:

        Scene_picker(Scene_renderer* renderer);

        bool select(Data_set& res, const Vec2i_cu &pos, const Params& p);

        void transform(const Data_set& s, const Transfo& gtransfo);

        void save_transform(const Data_set& set,
                            std::vector<Data_tr>& transfo);

        void load_transform(const std::vector<Data_tr>& transfo);

        /// Set a group of data selected last element is active
        /// @param s : wether we select or unselect
        void set_selected(const Data_set& set, bool s);

        void set_selected_all(bool state);

        void set_parent(const Data_set& set, Data* parent);

        void paste(Data_set_copy& /*set*/){ }

        void remove(const Data_set& s);

    private:
        /// shortcut to access scene tree
        Scene_tree* tree(){ return _renderer->_tree; }

        Scene_renderer* _renderer;
    };

    // -------------------------------------------------------------------------

    /// @brief functor to draw objects with depth peeling for transparent objects
    class Peel_functor : public Peeler::Render {
    public:
        Peel_functor(const Scene_renderer* rndr,
                     const Render_context* ctx,
                     const std::vector<Obj*>& objects ) :
            Peeler::Render(),
            _renderer( rndr ),
            _ctx(ctx),
            _objs(objects)
        {  }

        void draw_transc_objs()
        {
            const Transfo eye = _renderer->_cam->get_eye_transfo();
            for(unsigned i = 0; i < _objs.size(); ++i)
                _renderer->draw_object(_objs[i], eye);
        }

        const Scene_renderer*    _renderer;
        const Render_context*    _ctx;
        const std::vector<Obj*>& _objs;
    };


    // -------------------------------------------------------------------------
    /// @name Atributes
    // -------------------------------------------------------------------------

    Render_context* _ctx;

    const Camera* _cam;

    Scene_picker* _picker; ///< picker to select object into the scene
    Scene_tree*   _tree;   ///< scene to be rendered

    /// @name Scene objects sorted by rendering type:
    /// @{
    std::vector<Obj*> _raster_objs;    ///< plain renderering with raster
    std::vector<Obj*> _wires_objs;     ///< wireframe rendering with raster
    std::vector<Obj*> _peel_objs;      ///< depth peeling rendering
    std::vector<Obj*> _polygonize_objs;///< scalar field polygonisation rendering
    /// @}

    std::vector<Obj*> _selected_objs;

    /// Global transformation of the scene's objects
    std::map<const Obj*, Transfo> _global_tr;

    float _alpha_strength;
};

#endif // SCENE_RENDER_HPP__
