/*
 * This software is governed by the CeCILL-B license under French law and
 * abiding by the rules of distribution of free software.  You can  use, 
 * modify and/ or redistribute the software under the terms of the CeCILL-B
 * license as circulated by CEA, CNRS and INRIA at the following URL
 * "http://www.cecill.info" or the LICENCE.txt file present in this project.
*/

#include "scene_renderer.hpp"
#include "std_utils.hpp"
#include "glpick_fbo.hpp"
#include "marching_cubes.hpp"

// see scene_tree.hpp
// Implementation of the inner class Scene_picker from Scene_picker

Scene_renderer::Scene_picker::Scene_picker(Scene_renderer* renderer) :
    Picker(),
    _renderer(renderer)
{     }

// -----------------------------------------------------------------------------

bool Scene_renderer::Scene_picker::select(Data_set& res,
                                          const Vec2i_cu& pos,
                                          const Params& p)
{
    if( _renderer->_tree == 0) return false;

    _renderer->update_scene( _renderer->_tree );
    const Transfo eye = _renderer->_cam->get_eye_transfo();

    GlPick_FBO* pick = p._fbo_picker;
    std::vector<Obj*> obj_list( tree()->size() );

    Obj::Use_shader save_shaders(false);
    Transfo mat = p.cam.get_proj_transfo() * p.cam.get_eye_transfo();
    pick->begin(pos.x, pos.y, mat.transpose().m );
    {
        GLEnabledSave lighting(GL_LIGHTING, true, false);
        GLEnabledSave tex2D(GL_TEXTURE_2D, true, false);
        GLEnabledSave save_color_mat(GL_COLOR_MATERIAL, true, false);

        Scene_tree::const_iterator it = tree()->begin();
        for(int i = 0; it != tree()->end(); ++it, ++i)
        {
            Obj* object = *it;
            obj_list[i] = object;

            if( !object->state(Obj::SELECTABLE) ) break;
            Obj::Save_state save(object, Obj::SELECTABLE|Obj::RENDER);
            pick->set_name(i);

            glPushMatrix();
            if( object->type_object() == EObj::IMPLICIT &&
                object->type_render() == Obj::POLYGONIZE )
            {
                const Transfo tr = _renderer->_global_tr.find(object)->second;
                glAssert( glLoadMatrixf( (eye * tr).transpose().m ) );
                Marching_cubes::render_scalar_field_cpu( dynamic_cast<Node_implicit_surface*>(object), Vec3i_cu(16) );
                _renderer->draw_object( object, eye);
            }
            else if( object->type_render() != Obj::RAYTRACE )
            {
                _renderer->draw_object( object, eye);
            }
            glPopMatrix();
        }
    }
    int id = pick->end()._name;

    res.clear();
    if( id > -1) res.insert( obj_list[id] );

    return !res.empty();
}

// -----------------------------------------------------------------------------

static void transfo_functor(const Data_set& s,
                            const Transfo& gtransfo,
                            const Scene_tree* tree,
                            void (*apply)(Obj*, const Transfo& tr))
{
    std::list<Obj*> valid_objs;
    Data_set::const_iterator it_set = s.begin();
    for(; it_set != s.end(); ++it_set)
    {
        if( (*it_set)->type_data() == EData::OBJECT )
            valid_objs.push_back( (Obj*)*it_set );
        else
            assert(false); // This picker only handle Data_set of type Obj
    }

    std::list<Obj*> top_lvl_objs;
    tree->top_level_objs(valid_objs, top_lvl_objs);

    std::list<Obj*>::iterator it = top_lvl_objs.begin();
    for(; it != top_lvl_objs.end(); ++it)
        apply(*it, gtransfo );

}

static void transfo_fun(Obj* o, const Transfo& tr){ o->transform(tr); }

// -----------------------------------------------------------------------------

void Scene_renderer::Scene_picker::transform(const Data_set& s,
                                             const Transfo& gtransfo)
{
    transfo_functor(s, gtransfo, tree(), transfo_fun);
}

// -----------------------------------------------------------------------------

void Scene_renderer::Scene_picker::save_transform(const Data_set& s,
                                                  std::vector<Data_tr>& transfo)
{
    Data_set::const_iterator it_set = s.begin();
    transfo.clear();
    transfo.reserve( s.size() );
    for(; it_set != s.end(); ++it_set)
    {
        if( (*it_set)->type_data() == EData::OBJECT )
        {
            Obj* o = (Obj*)(*it_set);
            transfo.push_back( std::make_pair(o, o->local_frame()) );
        }else
            assert(false); // This picker only handle Data_set of type Obj
    }
}

// -----------------------------------------------------------------------------

void Scene_renderer::Scene_picker::load_transform(
        const std::vector<Data_tr>& transfo)
{
    for(unsigned i = 0; i < transfo.size(); ++i) {
        Obj* o = (Obj*)(transfo[i].first);
        o->set_local_transform( transfo[i].second );
    }
}

// -----------------------------------------------------------------------------

static void select_func(Data* d, bool s)
{
    if( d->type_data() == EData::OBJECT )
    {
        Obj* obj = (Obj*)d;
        obj->_selected = s;
    }else{
        assert(false);
    }
}

void Scene_renderer::Scene_picker::set_selected(const Data_set& set, bool s)
{
    Picker::set_selected_fun(set, s, select_func);
}

// -----------------------------------------------------------------------------

void Scene_renderer::Scene_picker::set_selected_all(bool state)
{
    if(!state){
        if(_selection.empty()) return;
        _selection.clear();
    }

    Scene_tree::const_iterator it = tree()->begin();
    for(; it != tree()->end(); ++it)
    {
        Obj* object = *it;
        if( object->type_render() != Obj::RAYTRACE &&
            object->state(Obj::SELECTABLE) )
        {
            object->_selected = state;
            if(state) _selection.insert(*it);
        }
    }
}

// -----------------------------------------------------------------------------

void Scene_renderer::Scene_picker::set_parent(const Data_set& s, Data* pt_)
{
    Obj* parent = (Obj*)pt_;
    // build a list of valid objects -> must not be the parent and subtree of
    // obj must not contain the parent
    std::list<Obj*> valid_objs;
    Data_set::const_iterator it_set = s.begin();
    for(; it_set != s.end(); ++it_set)
    {
        if( (*it_set)->type_data() == EData::OBJECT )
        {
            Obj* obj = (Obj*)(*it_set);

            if( !tree()->has(parent, /* from: */obj) && (obj != parent) )
                    valid_objs.push_back( obj );
        } else {
            assert( false );
        }
    }

    std::list<Obj*> top_lvl_objs;
    tree()->top_level_objs(valid_objs, top_lvl_objs);

    // Assign new parent to top level nodes
    std::list<Obj*>::iterator it_top = top_lvl_objs.begin();
    for( ; it_top != top_lvl_objs.end(); ++it_top)
        tree()->move_subtree(*it_top, parent);
}

// -----------------------------------------------------------------------------

static Scene_tree* g_remove_func_attr_tree = 0;
static void remove_func(Data* data)
{
    if( data->type_data() == EData::OBJECT )
    {
        Obj* obj = (Obj*)(data);
        g_remove_func_attr_tree->unregister_obj( obj );
        delete obj;
    }
}

void Scene_renderer::Scene_picker::remove(const Data_set& set)
{
    g_remove_func_attr_tree = tree();
    Picker::remove_fun( set, remove_func);
    g_remove_func_attr_tree = 0;
}

// -----------------------------------------------------------------------------
