/*
 * This software is governed by the CeCILL-B license under French law and
 * abiding by the rules of distribution of free software.  You can  use, 
 * modify and/ or redistribute the software under the terms of the CeCILL-B
 * license as circulated by CEA, CNRS and INRIA at the following URL
 * "http://www.cecill.info" or the LICENCE.txt file present in this project.
*/

#ifndef DATA_SET_HPP__
#define DATA_SET_HPP__

#include <set>

#include "transfo.hpp"
#include "data_enum.hpp"

// =============================================================================

/**
 * @class Data
 * @brief Base class for any selectable object in the 3D scene
 */
class Data {
public:
    virtual ~Data(){}
    /// Object frame, (for face or vertex aligned with it's normal)
    /// @warning the frame is expressed in global coordinates despite the fact
    /// it represent the local orientation of the object/face/vertex/whatever
    virtual Transfo frame() const = 0;

    /// @return the data type
    /// @see EData::Data_t
    virtual EData::Data_t type_data() const = 0;

    /// This acts as a virtual copy constructor.
    /// Any object can be duplicated as long as this method is correctly
    /// specialized
    virtual Data* clone() const = 0;
};

// =============================================================================

/**
 * @class Data_set
 * @brief a set of pointers
 */
class Data_set {
public:
    typedef std::set<Data*> Container;

    Data_set(){ }

    /// add without doubles
    void insert(const Data_set& s)
    {
        const_iterator it = s.begin();
        for(; it != s.end(); ++it)
            _collection.insert( *it );
    }

    /// add without doubles
    void insert(Data* s){ _collection.insert(s); }

    /// remove from collection
    void erase(const Data_set& s){
        const_iterator it = s.begin();
        for(; it != s.end(); ++it)
            _collection.erase( *it );
    }

    void erase(Data* s){ _collection.erase( s ); }

    /// remove from collection intersection then add what's left
    void exclusive_add(const Data_set& s)
    {
        const_iterator it = s.begin();
        for(; it != s.end(); ++it)
        {
            unsigned nb = _collection.erase( *it );
            if(nb == 0)
                _collection.insert( *it );
        }

    }

    /// flush the collection
    void clear(){ _collection.clear(); }

    /// @return wether the set is empty or not
    bool empty() const { return _collection.empty(); }

    int size() const { return (int)_collection.size(); }


    // -------------------------------------------------------------------------
    /// @name iterators
    // -------------------------------------------------------------------------
    typedef Container::iterator       iterator;
    typedef Container::const_iterator const_iterator;

    iterator begin(){ return _collection.begin(); }
    iterator end()  { return _collection.end();   }

    const_iterator begin() const { return _collection.begin(); }
    const_iterator end()   const { return _collection.end();   }

    typedef Container::reverse_iterator       riterator;
    typedef Container::const_reverse_iterator const_riterator;

    riterator rbegin(){ return _collection.rbegin(); }
    riterator rend()  { return _collection.rend();   }

    const_riterator rbegin() const { return _collection.rbegin(); }
    const_riterator rend()   const { return _collection.rend();   }

    // -------------------------------------------------------------------------
    /// @name Attributes
    // -------------------------------------------------------------------------
private:
    Container _collection;
};

// =============================================================================

// TODO: not sure we really need this class and makes things cumbersome by
// bloating the code
/**
 * @brief Handling memory copy of a data set
 */
class Data_set_copy {
public:

    Data_set_copy() {}

    /// All referenced copied memory is dropped at destruction
    ~Data_set_copy() { clear(); }

    /// delete all referenced copy
    void clear(){
        Data_set::iterator it = _set.begin();
        for(; it != _set.end(); ++it)
            delete (*it);
        _set.clear();
    }

    /// This really duplicates datas and allocate them.
    void copy(const Data_set& s)
    {
        Data_set::const_iterator it = s.begin();
        for(; it != s.end(); ++it)
        {
            const Data* elt_ptr = *it;
            _set.insert( elt_ptr->clone() );
        }
    }

    /// retreive referenced copy
    /// @warning caller will have to take care of the deletion of the data set
    /// 'copy' !
    void retreive(Data_set& copy){
        copy.clear();
        copy = _set;
        _set.clear();
    }

private:
    Data_set _set;
};

#endif // #define DATA_SET_HPP__
