/*
 * This software is governed by the CeCILL-B license under French law and
 * abiding by the rules of distribution of free software.  You can  use, 
 * modify and/ or redistribute the software under the terms of the CeCILL-B
 * license as circulated by CEA, CNRS and INRIA at the following URL
 * "http://www.cecill.info" or the LICENCE.txt file present in this project.
*/

#ifndef OBJECT_HPP__
#define OBJECT_HPP__

#include <vector>
#include "transfo.hpp"
#include "render_context.hpp"
#include "picker.hpp"
#include "object_enum.hpp"

// =============================================================================

class Scene_tree;

/**
 * @brief Base class for any objects of the scene
 *
 */
class Obj : public Data {
public:

    // -------------------------------------------------------------------------
    /// @name enums
    // -------------------------------------------------------------------------

    /// @brief Bit flags to store different object rendering states
    enum State_t {
        RENDER     = 1,  ///< Render object enable
        SELECTABLE = 2,  ///< Enable object selection
        WIRES      = 4,  ///< show object wires
        LIGHTING   = 8,  ///< compute lighting on object
        MATERIALS  = 16, ///< Use object material
        TEXTURE    = 32  ///< Use object texture
    };

    /// @brief Bit field of type State_t
    typedef int State_flags;

    /// @brief render type specifies which algorithm is to be used to render
    /// the object
    enum Render_t {
        PEEL       = 1,  ///< Render with depth peeling (transcluscent objects)
        RAYTRACE,        ///< Raytrace object (only implicit objects)
        POLYGONIZE,      ///< polygonize object (only implicit objects)
        RASTER           ///< Use standard opengl rasterization
    };

    // -------------------------------------------------------------------------
    /// @name methods
    // -------------------------------------------------------------------------

    Obj() :
        _selected(false),
        _frame_lcl( Transfo::identity() ),
        _parent(0),
        _state(RENDER|SELECTABLE|LIGHTING),
        _render_type(RASTER),
        _mode(EObj::MODE_OBJECT),
        _mode_list(EObj::MODE_OBJECT)
    {}

    virtual ~Obj(){ }

    virtual void draw() = 0;

    // -------------------------------------------------------------------------
    /// @name Getters & Setters
    // -------------------------------------------------------------------------

    virtual EObj::Obj_t type_object() const = 0;

    // TODO: rename this to 'check_state'
    /// Check a specific state flag
    /// @return wether flag is active/inactive
    bool state(State_t s) const { return (_state & (int)s) != 0; }

    /// @return every state flags
    State_flags state() const { return _state; }

    void set_state(State_flags state){ _state = state; }

    void set_state(State_t state, bool s){
        const int tmp = (int)state;
        _state = s ? (_state | tmp) : (_state & (~tmp));
    }

    Render_t type_render() const { return _render_type; }

    EObj::Mode_t type_mode() const { return _mode; }

    EObj::Flags modes_supported() const { return _mode_list; }

    const Obj* parent() const { return _parent; }
    Obj* parent() { return _parent; }

    const std::vector<Obj*>& sons() const { return _sons; }
    std::vector<Obj*>& sons() { return _sons; }


    // -------------------------------------------------------------------------
    /// @name Object's position/manipulations
    // -------------------------------------------------------------------------

    /// @return the object's global frame
    /// (by looking up the tree from 'this' up to the root node)
    Transfo global_frame() const;

    /// @return the object's frame expressed in local coordinates of its parent
    Transfo local_frame() const { return _frame_lcl; }

    /// Apply transformation to the object
    /// @param transfo : the transformation in <b>global coordinates<\b>
    void transform(const Transfo& transfo);

    /// Set transformation to the object
    /// @param transfo : the transformation in <b>global coordinates<\b>
    void set_transform(const Transfo& transfo);

    void set_local_transform(const Transfo& transfo) { _frame_lcl = transfo; }

    // -------------------------------------------------------------------------
    /// @name Data overloaded methods
    // -------------------------------------------------------------------------

    EData::Data_t type_data() const { return EData::OBJECT; }

    /// @return object's frame in global coordinates
    Transfo frame() const { return global_frame(); }

    /// Duplicate object into a newly allocated instance
    virtual Data* clone() const = 0;

    // -------------------------------------------------------------------------
    /// @name Inner class tools
    // -------------------------------------------------------------------------

    /// @brief utility to save/change and automatically restore shader state.
    struct Use_shader{
        Use_shader(bool s){  _save = Obj::_use_shader; Obj::_use_shader = s; }
        ~Use_shader(){ Obj::_use_shader = _save; }
        bool _save;
    };

    /// @brief utility to save/change and automatically restore object state.
    struct Save_state {
        Save_state(Obj* o, State_flags new_state)
        {
            _o = o;
            _save = o->state();
            o->set_state(new_state);
        }
        ~Save_state(){ _o->set_state( _save ); }
        State_flags _save;
        Obj* _o;
    };

    // -------------------------------------------------------------------------
    /// @name Attributes
    // -------------------------------------------------------------------------

    static bool _use_shader;

    bool        _selected;     ///< is the object selected
    std::string _name;         ///< Object's name

protected:
    friend class Scene_tree;

    /// Object position relative to its parent
    Transfo _frame_lcl;

    Obj*              _parent; ///< Object's parent
    std::vector<Obj*> _sons;   ///< Object's sons

    State_flags  _state;       ///< bit flags of type State_t
    Render_t     _render_type; ///< Specify the rendering algorithm
    EObj::Mode_t _mode;        ///< object's mode defines it's I/O behaviour
    EObj::Flags  _mode_list;   ///< suported modes for the I/O
};

#endif // OBJECT_HPP__
