/*
 * This software is governed by the CeCILL-B license under French law and
 * abiding by the rules of distribution of free software.  You can  use, 
 * modify and/ or redistribute the software under the terms of the CeCILL-B
 * license as circulated by CEA, CNRS and INRIA at the following URL
 * "http://www.cecill.info" or the LICENCE.txt file present in this project.
*/

#ifndef PICKER_HPP__
#define PICKER_HPP__

#include "picker_io.hpp"
#include "data_set.hpp"
#include "transfo.hpp"
#include "camera.hpp"
#include "vec3_cu.hpp"
#include "vec2i_cu.hpp"

#include <vector>

class GlPick_FBO;

/**
 * @class Picker
 * @brief Base class for item picking (scene objects, points, face edges etc.
 * Whatever obj_data to be moved/manipulated/selected)
 *
 * In order to factorize code related to input/output events every 3D elements
 * must implement this abstract class. A mesh, implicit tree, vertex, face etc.
 * usually are manipulated/moved/suppressed/selected/... the same way. There
 * is no point to redefine mouse and keyboard events for each type of Data.
 *
 * Picker class intend to factorize these events handling by providing
 * this abstract interface. It allows IO events to be dealt with generically for
 * any type of Data which re-implement this class.
 *
 * @note Take a look in the section Tools of this class. You'll find functions
 * to help you to quickly override virtual functions.
 * @warning read carefully the documentation of each method and follow
 * it strictly in order to avoid discrepancies in the code.
*/
class Picker {
public:
    Picker( Picker_io* ext = 0 ) : _selection(), _active(0), _io_extension(ext) { }

    virtual ~Picker(){}

    /// @brief Parameters for selection
    struct Params {
        Camera cam;
        GlPick_FBO* _fbo_picker;
        // TODO:
        // heuristic circle/mouse etc ...
    };

    // =========================================================================
    /// @name Moving the object
    // =========================================================================

    /// Apply a transformation to a set of data
    /// @param set : the data set to transform from it's initial position
    /// @param transfo : transformation to apply to the data set in <b>global
    /// coordinates<\b>
    virtual void transform(const Data_set& set, const Transfo& transfo) = 0;

    typedef std::pair<Data*, Transfo> Data_tr;

    /// Save transformation in 'set' to 'transfo'. they can be reloaded later
    /// with load_transform()
    /// @warning the format of transfo must be consistent: save_transform()
    /// is solely meant to provide transformation to load_transform()
    virtual void save_transform(const Data_set& set,
                                std::vector<Data_tr>& transfo) = 0;

    /// Load transformations from 'transfo' vector build from with a call
    /// to save_transform()
    /// @warning the format of transfo must be consistent: load_transform()
    /// is solely meant to receive transformations from load_transform()
    virtual void load_transform(const std::vector<Data_tr>& transfo) = 0;

    // =========================================================================
    /// @name Selecting objects
    // =========================================================================

    /// Select an object in the scene with the mouse.
    /// @param res : the set of datas selected
    /// @param pos : mouse position (qt origin: upper left corner)
    /// @param p : parameters as the buffers, camera position, area selected...
    virtual bool select(Data_set& res, const Vec2i_cu& pos, const Params& p) = 0;

    /// Set a group of data selected last element is active
    /// @param s : wether we select or unselect
    /// @note if s == true it does not unselect previously selected data.
    /// Use set_selected_all(false); to unselect previously selected data
    /// @note you can use set_selected_fun() to implement it.
    virtual void set_selected(const Data_set& set, bool s) = 0;

    /// Select or deselect everything (more efficient than set_selected())
    /// @param state wether we select or deselect.
    virtual void set_selected_all(bool state) = 0;

    // =========================================================================
    /// @name Copy/Paste/Remove datas
    // =========================================================================

    /// Paste data
    /// @param set : data set to be added to the existing scene.
    /// Pasting will clear the buffer
    virtual void paste(Data_set_copy& set) = 0;

    /// remove data set 's' from the group
    /// @warning the set will point to erased memory don't forget to empty it.
    /// @warning You must unselect the set before removing it
    /// @note use remove_fun() to implement it
    virtual void remove(const Data_set& s) = 0;

    //TODO: Joins a set of objects of the same type into one.
    // Data_set is left unchanged you must remove it
    //virtual Data* join(const Data_set& s) = 0;

    // =========================================================================
    /// @name Miscellaneous
    // =========================================================================

    // TODO: maybe this function should only be present for the
    // Scene_tree::Picker, The other types of objects doesn't need to set
    // parent relations
    /// Sets the parent of a group of data.
    /// @param set : the set to be attributed with a new parent.
    /// @param parent : the new parent to attribute to 'set'
    /// @note the set may include or not 'parent'.
    /// local frame of the objects will be updated acording to its new parent
    virtual void set_parent(const Data_set& set, Data* parent) = 0;

    /// Only one data element can be active at the same time
    virtual void set_active(Data* d){ _active = d; }

    /// When the picker becomes the active IO this method is called
    /// Typical action to be done: unhide mesh edges/vertices for edition
    virtual void hold() { }

    /// When the picker is no the active IO anymore this method is called
    /// Typical action to be done: hide mesh's wires, uncolor what's selected...
    virtual void release() { }

    // =========================================================================
    /// @name Accessors
    // =========================================================================

    /// Wether there is an active element in the selection
    bool has_active() const { return _active != 0; }

    /// @return active data or NULL if no active data
    Data* get_active() const { return _active; }

    const Data_set& get_selected() const { return _selection; }

    /// @return the IO extension specifically defined for the object or
    /// NULL if no spezialised IO is required.
    Picker_io* get_io_extension() const { return _io_extension; }

    // =========================================================================
    /// @name Tools
    /// Abstract methods will often behave the same way use the following
    /// functor to factorize your code !
    // =========================================================================

    /// A functor to help factorizing code in 'set_selected()'
    /// @param set : the set to un/select (depending on 's' value)
    /// @param s : select or unselect ?
    /// @param select_data : your function to customize selection
    /// it will be called for every elements in 'set' with the boolean equal to
    /// 's'
    void set_selected_fun(const Data_set& set,
                          bool s,
                          void (*select_data)(Data*, bool) )
    {
        Data_set::const_iterator it = set.begin();
        for(; it != set.end(); ++it)
        {
            select_data( *it, s);
            if(s)
                _selection.insert(*it);
            else
            {
                if( *it == _active ) set_active( 0 );
                _selection.erase(*it);
            }
        }
    }

    /// A functor to help factorizing  code in 'remove()'
    void remove_fun(const Data_set& set, void (*remove_data)(Data*))
    {
        Data_set::const_iterator it = set.begin();

        for(; it != set.end(); ++it)
        {
            if( *it == _active ) set_active( 0 );

            remove_data(*it);
        }
    }


protected:
    Data_set   _selection;     ///< currently selected data
    Data*      _active;        ///< current active data
    Picker_io* _io_extension;  ///< Interface to define a specific IO behaviour
};

#endif // PICKER_HPP__
