/*
 * This software is governed by the CeCILL-B license under French law and
 * abiding by the rules of distribution of free software.  You can  use, 
 * modify and/ or redistribute the software under the terms of the CeCILL-B
 * license as circulated by CEA, CNRS and INRIA at the following URL
 * "http://www.cecill.info" or the LICENCE.txt file present in this project.
*/

#include "scene_tree.hpp"

#include "assert_utils.hpp"
#include "std_utils.hpp"

// -----------------------------------------------------------------------------

/// @return true if a cycle is detected
static bool rec_is_cycles(std::set<const Obj*>& seen, const Obj* curr)
{
    // Already inserted it's a cycle
    if( !seen.insert(curr).second )
        return true;

    bool cycle = false;
    for(unsigned i = 0; i < curr->sons().size(); ++i)
        cycle = cycle || rec_is_cycles(seen, curr->sons()[i]);

    return cycle;
}

/// @return true if a cycle is detected
static bool is_cycles(const Obj* start)
{
    std::set<const Obj*> seen_buff;
    return rec_is_cycles(seen_buff, start);
}

// -----------------------------------------------------------------------------

void Scene_tree::transfer_tree(Scene_tree& ext_tree, Obj* parent)
{
    if( parent == 0) parent = _root;
    assert_msg( has(parent), "ERROR: parent never registered in this tree." );
    assert_msg( check()    , "ERROR: corrupted tree detected."              );


    Transfo to_parent_lcl = parent->global_frame().fast_invert();

    // Connect external tree to the current tree
    Obj* ext_root = ext_tree._root;
    for(unsigned i = 0; i < ext_root->_sons.size(); ++i)
    {
        Obj* son = ext_root->_sons[i];
        // Update matrix
        son->_frame_lcl = to_parent_lcl * ext_root->_frame_lcl * son->_frame_lcl;
        // Link objects
        parent->_sons.push_back( son );
        son->_parent = parent;
    }


    // Update object list
    iterator it = ext_tree._obj_list.begin();
    for(; it != ext_tree._obj_list.end(); ++it)
        _obj_list.push_back( *it );

    ext_tree._obj_list.clear();

}

// -----------------------------------------------------------------------------

void Scene_tree::register_obj(Obj* obj, Obj* parent)
{
    if( parent == 0) parent = _root;
    assert_msg( !has(obj)  , "ERROR: object already registered in the tree." );
    assert_msg( has(parent), "ERROR: parent never registered in this tree."  );
    assert_msg( check()    , "ERROR: corrupted tree detected."               );
    assert_msg(obj->_sons.size() == 0, "ERROR: object is not a leaf."        );
    assert_msg(obj->_parent == 0     , "ERROR: object is not root"           );

    Transfo gtransfo = obj->global_frame();

    obj->_parent = parent;
    parent->_sons.push_back( obj );
    obj->set_transform( gtransfo );

    _obj_list.push_back( obj );
}

// -----------------------------------------------------------------------------

/// remove every objects in the subtree starting from 'curr' in the vector
/// tree_objs if they exists.
static void rec_remove_from_list(std::vector<Obj*>& tree_objs,
                                 Obj* curr)
{
    assert_msg( Std_utils::exists(tree_objs, curr),
                "ERROR: some of the node of the subtree are not registered.");

    Std_utils::erase( tree_objs, Std_utils::find(tree_objs, curr) );

    for(unsigned i = 0; i < curr->sons().size(); ++i)
        rec_remove_from_list(tree_objs, curr->sons()[i]);
}

// -----------------------

void Scene_tree::unregister_subtree(Obj* start)
{
    assert_msg( check(), "ERROR: corrupted tree detected." );
    assert_msg( start->_parent != 0, "ERROR: corrupted tree? object registered "
                                     "should always have a parent." );
    Obj* parent = start->_parent;
    assert_msg( has(parent), "ERROR: subtree not connected to tree." );
    assert_msg( Std_utils::exists( parent->sons(), start ),
                "ERROR: subtree not connected to tree." );

    // update matrices
    start->_frame_lcl = start->global_frame();

    // -----------------
    // Update obj PARENT
    // Delete from sons
    std::vector<Obj*>& psons = parent->_sons;
    Std_utils::erase( psons, Std_utils::find(psons, start) );

    // -------------
    // Update OBJECT
    start->_parent = 0;

    // Remove from the list of objects
    rec_remove_from_list(_obj_list, start);
}

// -----------------------------------------------------------------------------

void rec_add_to_list(std::vector<Obj*>& tree_objs, Obj* curr)
{
    assert_msg( !Std_utils::exists(tree_objs, curr),
                "ERROR: some of the node of the subtree are registered.");

    tree_objs.push_back( curr );
    for(unsigned i = 0; i < curr->sons().size(); ++i)
        rec_add_to_list( tree_objs, curr->sons()[i] );
}

// -----------------

void Scene_tree::register_subtree(Obj* start, Obj* parent)
{
    if( parent == 0) parent = _root;
    assert_msg( check()            , "ERROR: corrupted tree detected."       );
    assert_msg( start->_parent == 0, "ERROR: root object not have a parent." );
    assert_msg( has(parent)        , "ERROR: parent is not registered."      );

    start->_parent = parent;
    parent->_sons.push_back( start );

    // Compute transfo to go in local coordinates of parent
    Transfo to_parent_lcl = parent->global_frame().fast_invert();
    start->_frame_lcl = to_parent_lcl * start->_frame_lcl;

    rec_add_to_list( _obj_list, start );
}

// -----------------------------------------------------------------------------

void Scene_tree::move_subtree(Obj* root, Obj* parent)
{
    if( parent == 0) parent = _root;
    assert_msg( !has(parent, root), "ERROR: root must not be in the subtree." );

    unregister_subtree(root);
    register_subtree(root, parent);
}

// -----------------------------------------------------------------------------

void Scene_tree::move_obj(Obj* obj, Obj* parent)
{
    if( parent == 0) parent = _root;
    assert_msg( has(obj)   , "ERROR: object never registered in this tree."  );
    assert_msg( has(parent), "ERROR: parent never registered in this tree."  );
    assert_msg( check()    , "ERROR: corrupted tree detected."               );

    // disconnect the object from its old location
    unregister_obj( obj );
    // reconnect it to its new parent
    register_obj( obj, parent);
}

// -----------------------------------------------------------------------------

void Scene_tree::unregister_obj(Obj* obj)
{
    assert_msg( has(obj), "ERROR: object never registered in this tree." );
    assert_msg( check() , "ERROR: corrupted tree detected."              );

    Transfo gtransfo = obj->global_frame();

    // -----------------
    // Update obj PARENT
    std::vector<Obj*>::iterator it;
    if(obj->_parent != 0)
    {
        // Delete from sons
        std::vector<Obj*>& psons = obj->_parent->_sons;
        Std_utils::erase( psons, Std_utils::find(psons, obj) );

        // add object's sons to its parent
        it = obj->_sons.begin();
        for( ; it != obj->_sons.end(); ++it)
            psons.push_back( *it );
    }else
        assert_msg( false, "ERROR: corrupted tree ?"
                           "object registered should always have a parent" );

    // ---------------
    // Update obj SONS
    // Change object's sons  parents update matrix accordingly
    it = obj->_sons.begin();
    for( ; it != obj->_sons.end(); ++it){
        Obj* son_obj = *it;
        son_obj->_frame_lcl = obj->_frame_lcl * son_obj->_frame_lcl;
        (*it)->_parent = obj->_parent;
    }

    // ----------
    // Update OBJ
    obj->_parent = 0;
    obj->_sons.clear();
    obj->_frame_lcl = gtransfo;

    // Remove from the list of objects
    Std_utils::erase( _obj_list, Std_utils::find(_obj_list, obj) );
}

// -----------------------------------------------------------------------------

void Scene_tree::delete_obj(Obj* obj)
{
    unregister_obj( obj );
    delete obj;
}

// -----------------------------------------------------------------------------

static void rec_compute_global_tr(const Obj* o,
                                  const Transfo& tr,
                                  std::map<const Obj*, Transfo>& tr_map)
{
    Transfo res = tr * o->local_frame();
    tr_map[o] = res;
    for(unsigned i = 0; i < o->sons().size(); ++i)
        rec_compute_global_tr( o->sons()[i], res, tr_map);
}


void Scene_tree::compute_global_tr(const Obj* o,
                                   std::map<const Obj*, Transfo>& tr_map)
{
    assert_msg( !is_cycles( o ), "ERROR: corrupted tree detected.");
    rec_compute_global_tr(o, Transfo::identity(), tr_map);
}

// -----------------------------------------------------------------------------

void Scene_tree::get_objs(EObj::Obj_t type, std::vector<Obj*>& objects) const
{
    assert_msg(check(), "ERROR: corrupted tree detected.");

    objects.clear();
    const_iterator it = begin();
    for (; it != end(); ++it) {
        Obj* o = *it;
        if( o->type_object() == type)
            objects.push_back( o );
    }
}

// -----------------------------------------------------------------------------

/// from a set of nodes 'candidates' build a list composed uniquely
/// from the higher nodes in the tree ( stored in 'top_lvl_objs' list).
/// @return wether a top level node has been found
static bool rec_build_top_node_list(Obj* node,
                                    std::set <Obj*>& candidates,
                                    std::list<Obj*>& top_lvl_objs )
{
    std::set<Obj*>::iterator it = candidates.find( node );
    if( it != candidates.end() )
    {
        top_lvl_objs.push_back( node );
        candidates.erase( it );
        return true;
    }

    bool state = false;
    for(unsigned i = 0; i < node->sons().size(); ++i)
    {
        bool res = rec_build_top_node_list( node->sons()[i],
                                            candidates,
                                            top_lvl_objs );
        state = state || res;
    }
    return state;
}

// -----------------------------------------------------------------------------

bool Scene_tree::top_level_objs(const std::list<Obj*>& candidates,
                                std::list<Obj*>& top_objs,
                                Obj* start) const
{
    if(start == 0) start = _root;
    assert_msg( check(), "ERROR: corrupted tree detected.");

    std::set<Obj*> candidate_set;
    std::list<Obj*>::const_iterator it = candidates.begin();
    for(; it != candidates.end(); ++it){
        assert_msg( has( *it ), "ERROR: candidate object never registered.");
        candidate_set.insert( *it );
    }

    return rec_build_top_node_list(start, candidate_set, top_objs);
}

// -----------------------------------------------------------------------------

/// Utility for 'has()' methods to recursively look up the tree from
/// 'start' node and add apply 'check( obj )' to each node
/// @return true as soon as 'check( obj )' returns true or false if not
static bool rec_has(const Obj* start, bool (*check)(const Obj*) )
{
    if( check(start) ) return true;

    for(unsigned i = 0; i < start->sons().size(); ++i)
        if( rec_has( start->sons()[i], check) )
            return true;

    return false;
}

// -----------------------------------------------------------------------------

static EObj::Obj_t is_equal_type_obj = EObj::NONE;

bool is_equal_type(const Obj* curr){
    return curr->type_object() == is_equal_type_obj;
}

bool Scene_tree::has(EObj::Obj_t object_type , const Obj* start) const
{
    assert_msg( has(start), "ERROR: object never registered in this tree." );
    assert_msg( check()   , "ERROR: corrupted tree detected."              );

    is_equal_type_obj = object_type;
    bool res = rec_has( start != 0  ? start : _root, is_equal_type);
    is_equal_type_obj = EObj::NONE;
    return res;
}

// -----------------------------------------------------------------------------

static Obj* is_equal_ptr_obj = 0;

bool is_equal_ptr(const Obj* curr){
    return curr == is_equal_ptr_obj;
}

bool Scene_tree::has(const Obj* obj, const Obj* start) const
{
    assert_msg( check(), "ERROR: corrupted tree detected.");

    is_equal_ptr_obj = const_cast<Obj*>( obj );
    bool res = rec_has(start != 0  ? start : _root, is_equal_ptr);
    is_equal_ptr_obj = 0;
    return res;
}

// -----------------------------------------------------------------------------

// recursively count the number of objects in the tree
static int rec_count_nodes(const Obj* start)
{
    int sub = 0;
    for(unsigned i = 0; i < start->sons().size(); ++i)
        sub += rec_count_nodes( start->sons()[i]);

    return sub + 1;
}

// -----------------------------------------------------------------------------

bool Scene_tree::check() const
{
    bool res = false;
    if( !is_cycles(_root) )
        res = rec_count_nodes( _root ) == ((int)_obj_list.size() + 1);

    return res;
}

// -----------------------------------------------------------------------------
