/*
 * This software is governed by the CeCILL-B license under French law and
 * abiding by the rules of distribution of free software.  You can  use, 
 * modify and/ or redistribute the software under the terms of the CeCILL-B
 * license as circulated by CEA, CNRS and INRIA at the following URL
 * "http://www.cecill.info" or the LICENCE.txt file present in this project.
*/

#ifndef SCENE_TREE_HPP__
#define SCENE_TREE_HPP__

#include "object.hpp"
#include "obj_dummy.hpp"

#include <vector>
#include <list>

/**
 * @brief Scene graph handling (adding/supressing/...)
 *
 * This class manage the object hierachy into the scene and memory.
 *
 * When the user registers/unregisters objects the class will ensure consistent
 * connectivity of the graph to keep a tree hierachy. Memory deletion of
 * registered objects will be performed as well. Transformation matrices
 * relations between objects will be kept consistent (i.e object frame will be
 * kept local to their parent at all time)
 *
 * Scene_tree provide the root node of the tree (see #root()):
 * objects registered are guaranted to have a parent node.
 *
 * One can add/remove objects to the tree like this:
 * @code
 * {
 *     Scene_tree tree;
 *     Obj* obj0 = new Obj_mesh();
 *     Obj* obj1 = new Obj_skeleton();
 *     tree.register(obj0);       // add obj0 at the top level of the tree
 *     tree.register(obj1, obj0); // add obj1 with parent set to be obj0
 *
 *     tree.unregister(obj0);
 *     // because obj0 has been unregistered we must perform the memory deletion
 *     // ourselves
 *     delete obj0;
 *
 * } // destruction of Scene_tree and its the registered objects (i.e. obj1)
 * @endcode
 *
 * @see Obj
*/
class Scene_tree {
public:

    // TODO: might want to use a std::set instead for efficiency
    typedef std::vector<Obj*> Container;

    Scene_tree() {
        _root = new Obj_dummy();
        _root->_name = "root";
        _root->set_state(0);
    }

    /// delete root node and delete every registered objects by add_obj()
    /// unless it has been unregisterd by delete_obj()
    ~Scene_tree() {
        for (unsigned i = 0; i < _obj_list.size(); ++i)
            delete _obj_list[i];

        delete _root;
    }

    // -------------------------------------------------------------------------
    /// @name Manipulate tree nodes (add/supr/concat/...)
    // -------------------------------------------------------------------------

    /// Transfer a sub tree to the scene. if no parent is specified add it to
    /// the top level of the hierachy
    /// @param tree : the hierachy to transfer.
    /// @warning 'tree' will be cleared in the process and all objects will be
    /// registered in the tree it had been transfered to. Object destructions
    /// will be handled by the tree they now belong to.
    void transfer_tree(Scene_tree& tree, Obj* parent = 0);

    /// Add an object to the tree. if no parent is specified adds it to the
    /// top level of the hierachy.
    /// @warning the object is now registered by the tree and
    /// will be deleted at its destruction
    /// @warning obj must be singleton (no leaf no parent). local frame must
    /// be equal to global
    /// (use register_subtree() to register all descendant at the same time)
    void register_obj(Obj* obj, Obj* parent = 0);

    /// Move an object to a new location in the tree. Its new parent will be
    /// set to 'parent' or '_root' if the later is set to null matrices are
    /// updated acordingly
    void move_obj(Obj* obj, Obj* parent = 0);

    /// unregister 'obj' and remove it from the tree.
    /// its parent will be connected to its children.
    /// children matrices will be updated and 'obj' local_frame() will be equal
    /// to its global_frame().
    /// @warning memory deletion is left to the user
    void unregister_obj(Obj* obj);

    /// Same as unregister_obj() but deletes 'obj' as well
    void delete_obj(Obj* obj);

    /// register 'obj' and all its descendants
    /// @warning the hierachy is now registered by the tree and
    /// will be deleted at its destruction
    void register_subtree(Obj* root, Obj* parent = 0);

    /// unregister 'obj' and all its descendants from the tree.
    /// its parent will be connected to its children.
    /// children matrices will be updated as well as the subtree matrices.
    /// @warning memory deletion is left to the user
    void unregister_subtree(Obj* root);

    /// Move an object and its descendants to a new location in the tree.
    /// Its new parent will be set to 'parent' or '_root' if the later is set
    /// to null matrices are updated acording to the new hierachy
    void move_subtree(Obj* root, Obj* parent = 0);

    // TODO: move this to a tool header to make the interface of the class lighter
    // TODO: set the object name and garantee its uniqueness in the tree
    // a number '.xx' is added at the end to avoid duplicates
    // void set_obj_name(Obj* obj, const std::string& name)

    // -------------------------------------------------------------------------
    /// @name Operations on frames
    // -------------------------------------------------------------------------

    // TODO: move this to a tool header to make the interface of the class lighter
    /// Compute global transformations of the subtree starting from 'obj_start'
    /// @param tr_map : list of global transformations of objects belonging to
    /// the subtree 'obj_start'
    static void compute_global_tr(const Obj* obj_start,
                                  std::map<const Obj*, Transfo>& tr_map);

    // -------------------------------------------------------------------------
    /// @name tests
    // -------------------------------------------------------------------------

    /// @param start : the sub tree we look up for the search
    /// (if null -> look up the whole tree)
    /// @return true if one occurence of the object type is found
    bool has( EObj::Obj_t object_type, const Obj* start = 0) const;

    /// @param start : the sub tree we look up for the search
    /// (if null -> look up the whole tree)
    /// @return true if one occurence of the object pointer is found
    bool has(const Obj* obj, const Obj* start = 0) const;

    /// @return false if the tree is inconsistent: for instance if cycles are
    /// detected or disconnected components
    bool check() const;

    // -------------------------------------------------------------------------
    /// @name getters
    // -------------------------------------------------------------------------

    /// Get the list of objects matching 'type' in the scene tree
    void get_objs(EObj::Obj_t type, std::vector<Obj*>& objects) const;

    // TODO: move this to a tool header to make the interface of the class lighter
    /// from a list of node 'candidates' build a list composed uniquely
    /// from the higher nodes in the tree (stored in 'top_objs' list).
    /// @param start : the subtree we start looking for top level objs:
    /// objects higher than 'start' will not be added. If start is null we begin
    /// from 'root()'
    /// @return wether a top level node has been found
    bool top_level_objs(const std::list<Obj*>& candidates,
                        std::list<Obj*>& top_objs,
                        Obj* start = 0) const;

    /// @return Number of registered objects in the tree
    int size() const { return _obj_list.size(); }

    /// @return root object of this tree
    Obj* root() const { return _root; }

    // -------------------------------------------------------------------------
    /// @name iterators
    /// Use these iterators to look up every object stored in the tree
    /// @code
    ///     Scene_tree::iterator it = tree.begin();
    ///     for(; it != tree.end(); ++it){ (*it)->object_method() }
    /// @endcode
    // -------------------------------------------------------------------------
    typedef Container::iterator       iterator;
    typedef Container::const_iterator const_iterator;

    iterator begin(){ return _obj_list.begin(); }
    iterator end()  { return _obj_list.end();   }

    const_iterator begin() const { return _obj_list.begin(); }
    const_iterator end()   const { return _obj_list.end();   }

private:
    // -------------------------------------------------------------------------
    /// @name Attributes
    // -------------------------------------------------------------------------

    ///  List of every objects in the scene tree
    Container _obj_list;

    /// the tree's root does not represent any objects but just register as
    /// sons new objects without parents
    Obj* _root;
};

#endif // SCENE_TREE_HPP__
