/*
 * This software is governed by the CeCILL-B license under French law and
 * abiding by the rules of distribution of free software.  You can  use, 
 * modify and/ or redistribute the software under the terms of the CeCILL-B
 * license as circulated by CEA, CNRS and INRIA at the following URL
 * "http://www.cecill.info" or the LICENCE.txt file present in this project.
*/

#include "grid3_cu.hpp"
#include "grid2_cu.hpp"

#include <QTime>


/**
 * @file containers_unit_test.cpp
 * @brief some unit tests for containers module (mostly 3d / 2d grids etc)
 *
 */
// =============================================================================
namespace Containers {
// =============================================================================

#ifdef NDEBUG

void unit_test(){ }

#else
template <typename E>
void fill0( Grid2_ref<E> g2 ){
    Vec2i_cu s = g2.alloc_size();
    s += s;
    g2(0, 0) = -1;
}

// -----------------------------------------------------------------------------

template <typename E>
void fill1( Grid2_ref<E> g2 ){
    Vec2i_cu s = g2.alloc_size();
    s += s;
    g2(0, 1) = -2;
}

// -----------------------------------------------------------------------------

void ref_test()
{
    int tab[4] = {1, 2, 3, 4 };
    Grid2_cu<int> g2(Vec2i_cu(2), tab);

    Grid2_ref<int> tmp0( g2 );
    fill0( tmp0 );
    assert( g2  (0, 0) == -1 );
    assert( tmp0(0, 0) == -1 );
    g2 = Grid2_cu<int>(Vec2i_cu(2), tab);
    tmp0 = g2;
    assert( tmp0(0, 0) == 1 );
}

// -----------------------------------------------------------------------------

template <typename E>
void get0( Grid2_const_ref<E> g2, E& res){
    Vec2i_cu s = g2.alloc_size();
    s += s;
    res = g2(0, 0);
}

// -----------------------------------------------------------------------------

template <typename E>
void get1( const Grid2_const_ref<E>& g2, E& res){
    Vec2i_cu s = g2.alloc_size();
    s += s;
    res = g2(0, 1);
}

// -----------------------------------------------------------------------------

template <typename E>
void get2( const Grid2_ref<E>& g2, E& res){
    Vec2i_cu s = g2.alloc_size();
    s += s;
    res = g2(1, 0);
}

// -----------------------------------------------------------------------------

void unit_test()
{
    Grid3_cu<int> g3;
    Grid2_cu<int> g2;
    Grid2_cu<int> g2_;

    int tab[8] = {1, 2,
                  3, 4,

                  5, 6,
                  7, 8};

    // This check constructors and assignement operator
    g3  = Grid3_cu<int>(Vec3i_cu(2), tab);
    g2  = Grid2_cu<int>(Vec2i_cu(2), tab);
    g2_ = Grid2_cu<int>(Vec2i_cu(2), tab+4);

    assert( g3(0, 0, 0) == 1 && g3(1, 0, 0) == 2 &&
            g3(0, 1, 0) == 3 && g3(1, 1, 0) == 4 &&

            g3(0, 0, 1) == 5 && g3(1, 0, 1) == 6 &&
            g3(0, 1, 1) == 7 && g3(1, 1, 1) == 8);

    assert( g2(0, 0) == 1 && g2(1, 0) == 2 &&
            g2(0, 1) == 3 && g2(1, 1) == 4 );

    assert( g2_(0, 0) == 5 && g2_(1, 0) == 6 &&
            g2_(0, 1) == 7 && g2_(1, 1) == 8 );

    //----------------------------------------
    // Test reference Grid2_ref mechanism
    //----------------------------------------

    // A class object Grid2_cu should be automatically converted to an Grid2_ref
    fill0( g2 );
    assert( g2(0,0) == -1);

    ref_test();

    fill1<int>( g2_ );
    assert( g2_(0, 1) == -2 );

    Grid2_ref<int> tmp = g3(A, A, 1);
    fill1( tmp );
    assert( g3(0, 1, 1) == -2 );
    g3(0, 1, 1) = 7;

    assert( g3(0, 0, 1) == 5 );
    fill0( g3(A, A, 1) );
    assert( g3(0, 0, 1) == -1 );

    assert( g3(0, 1, 1) == 7 );
    fill1( g3(A, A, 1) );
    assert( g3(0, 1, 1) == -2 );
    g3(0, 0, 1) = 5;
    g3(0, 1, 1) = 7;

    // Since tmp is seen as a reference we expect assignement to do an
    // hard copy of g2 into what's is referenced i.e: a slice of g3
    tmp = g2;
    assert( tmp(0, 0) == -1 && tmp(1, 0) == 2 &&
            tmp(0, 1) ==  3 && tmp(1, 1) == 4 );

    assert( tmp(0, 0) == g2(0, 0) && tmp(1, 0) == g2(1, 0) &&
            tmp(0, 1) == g2(0, 1) && tmp(1, 1) == g2(1, 1) );

    assert( g3(0, 0, 1) == -1 && g3(1, 0, 1) == 2 &&
            g3(0, 1, 1) ==  3 && g3(1, 1, 1) == 4 );

    // assign
    g2_ = tmp;
    assert( g2_(0, 0) == -1 && g2_(1, 0) == 2 &&
            g2_(0, 1) ==  3 && g2_(1, 1) == 4 );

    assert( tmp(0, 0) == g2_(0, 0) && tmp(1, 0) == g2_(1, 0) &&
            tmp(0, 1) == g2_(0, 1) && tmp(1, 1) == g2_(1, 1) );

    assert( g2(0, 0) == -1 && g2(1, 0) == 2 &&
            g2(0, 1) ==  3 && g2(1, 1) == 4 );

    fill1( g3(A, A, 0) );
    g2 = g3(A, A, 0);
    assert( g2(0, 0) ==   1 && g2(1, 0) == 2 &&
            g2(0, 1) ==  -2 && g2(1, 1) == 4 );

    //----------------------------------------
    // Test matlab like accessors (non const)
    //----------------------------------------
    int tab1[8] = { 9, 10,
                    11, 12,

                    13, 14,
                    15, 16};

    g2 = Grid2_cu<int>(Vec2i_cu(2), tab1);

    g3(A, A, 0) = g2;

    assert( g3(0, 0, 0) ==  9 && g3(1, 0, 0) == 10 &&
            g3(0, 1, 0) == 11 && g3(1, 1, 0) == 12 );

    g3  = Grid3_cu<int>(Vec3i_cu(2), tab1);
    g3(A, A, 0) = g3(A, A, 1);

    assert( g3(0, 0, 0) == 13 && g3(1, 0, 0) == 14 &&
            g3(0, 1, 0) == 15 && g3(1, 1, 0) == 16 );

    // reset values
    Grid3_cu<int> g3_bis(Vec3i_cu(2), tab);
    g3_bis(A, A, 0) = g3(A, 1, A);

    assert( g3_bis(0, 0, 0) == 15 && g3_bis(1, 0, 0) == 16 &&
            g3_bis(0, 1, 0) == 15 && g3_bis(1, 1, 0) == 16 &&

            g3_bis(0, 0, 1) == 5 && g3_bis(1, 0, 1) == 6 &&
            g3_bis(0, 1, 1) == 7 && g3_bis(1, 1, 1) == 8 );

    g3_bis(A, A, 1) = g3(0, A, A);

    assert( g3_bis(0, 0, 0) == 15 && g3_bis(1, 0, 0) == 16 &&
            g3_bis(0, 1, 0) == 15 && g3_bis(1, 1, 0) == 16 &&

            g3_bis(0, 0, 1) == 13 && g3_bis(1, 0, 1) == 15 &&
            g3_bis(0, 1, 1) == 13 && g3_bis(1, 1, 1) == 15 );

    //----------------------------------------
    // Test matlab like accessors (const)
    //----------------------------------------

    g3  = Grid3_cu<int>(Vec3i_cu(2), tab);
    g2  = Grid2_cu<int>(Vec2i_cu(2), tab);
    g2_ = Grid2_cu<int>(Vec2i_cu(2), tab+4);

    int val = -1;
    get0( g2, val);
    assert( val == 1);

    get0( g2(A, A), val);
    assert( val == 1);

    get1( g2_, val);
    assert( val == 7);

    get0( g3(A, A, 0), val);
    assert( val == 1);

    get1( g3(A, A, 1), val);
    assert( val == 7);

    get2( g3(A, A, 1), val);
    assert( val == 6);

    //----------------------------------------
    // Test Range accessors
    //----------------------------------------
                 // First z slice
    int tab2[16] = {1,2,3,4,
                    5,6,7,8,
                    // Second z slice
                     9,10,11,12,
                    13,14,15,16};

    // This check constructors and assignement operator
    g3  = Grid3_cu<int>(Vec3i_cu(4, 2, 2), tab2);

    g2 = g3( Range(1,2), Range(0,1), 0);

    assert( g2(0, 0) == 2 && g2(1, 0) == 3 &&
            g2(0, 1) == 6 && g2(1, 1) == 7 );

    assert( Vec2i_cu(2) == g2.alloc_size() );

    g2 = g3( Range(1,2), 0, Range(0,1));

    assert( g2(0, 0) ==  2 && g2(1, 0) == 3 &&
            g2(0, 1) == 10 && g2(1, 1) == 11 );

    assert( Vec2i_cu(2) == g2.alloc_size() );

    g2 = g3( 3, Range(0,1), Range(0,1));

    assert( g2(0, 0) ==  4 && g2(1, 0) == 8 &&
            g2(0, 1) == 12 && g2(1, 1) == 16 );

    assert( Vec2i_cu(2) == g2.alloc_size() );

    //----------------------------------------
    // Perf test
    //----------------------------------------
    #if 0
    // Note: I found that the virtual funcs access are 4 to 5 times slower than
    // direct access... (In release mode)
    QTime t;

    // Test with virtual access
    t.start();
    int nb_iter = 1;
    int size = 128;
    Grid3_cu<int> large_g3( Vec3i_cu(size), 1);

    int acc = 0;
    for(int iter = 0; iter < nb_iter; ++iter){
        for(int i = 0; i < size; ++i) {
            Grid2_const_ref<int> ref_g2 = large_g3(A, i, A);
            for( Idx2_cu idx(Vec2i_cu(size), 0); idx.is_in(); ++idx )
                acc += ref_g2( idx );// This use a virtual func access
        }
    }

    std::cout << "look up grid with virtual func access: " << t.elapsed() << " ms" << std::endl;
    if( acc == size*size*size*nb_iter)
        std::cout << "Look up succeed" << std::endl;

    // Test without virtual access
    t.start();

    acc = 0;
    for(int iter = 0; iter < nb_iter; ++iter){
        for(int i = 0; i < size; ++i) {
            for( Idx2_cu idx(Vec2i_cu(size), 0); idx.is_in(); ++idx ){
                Vec2i_cu v = idx.to_2d();
                acc += large_g3( v.x, i, v.y); // This does not use virtual func
            }
        }
    }

    std::cout << "look up grid without virtual func access: " << t.elapsed() << " ms" << std::endl;
    if( acc == size*size*size*nb_iter)
        std::cout << "Look up succeed" << std::endl;
    #endif
}
#endif

}// END NAMESPACE CONTAINERS ===================================================
