/*
 * This software is governed by the CeCILL-B license under French law and
 * abiding by the rules of distribution of free software.  You can  use, 
 * modify and/ or redistribute the software under the terms of the CeCILL-B
 * license as circulated by CEA, CNRS and INRIA at the following URL
 * "http://www.cecill.info" or the LICENCE.txt file present in this project.
*/

#include "grid2_const_ref.hpp"

#include "grid3_cu.hpp"
#include "grid2_cu.hpp"

// -----------------------------------------------------------------------------
// Implemens of internal classes
// -----------------------------------------------------------------------------

/// @brief Abstract class to point to a 2d slice of a 3d grid
template<typename T>
struct Grid2_ref_slice : public Grid2_cu<T> {

    Grid2_ref_slice( const Grid2_ref_slice<T>& copy ) :
        Grid2_ref<T>( *this ),
        _grid_ref(copy._grid_ref),
        _slice( copy._slice ),
        _r0( copy._r0 ),
        _r1( copy._r1 )
    {
        Grid2_cu<T>::_size    = copy._size;
        Grid2_cu<T>::_pad_off = copy._pad_off;
    }

    Grid2_ref_slice( Grid3_cu<T>& g, int slice, Range r0, Range r1) :
        _grid_ref( g ),
        _slice(slice),
        _r0( r0 ),
        _r1( r1 )
    {
    }

    virtual       T& get_val(int, int )       = 0;
    virtual const T& get_val(int, int ) const = 0;

protected:
    // You must call this for each children ctor.
    void init_size(const Vec2i_cu& alloc_size, const Vec2i_cu& padd_off) {
        Grid2_cu<T>::_size    = alloc_size;
        Grid2_cu<T>::_pad_off = padd_off;
    }

    /// 3d grid reference for the 2d grid slice
    Grid3_cu<T>& _grid_ref;
    /// slice index in the 3d grid
    /// (x, y or z direction it depends on the child class interpretation)
    int _slice;
    /// Ranges are to be initialized by children classes
    Range _r0, _r1;
};


// -----------------------------------------------------------------------------

template<typename T>
struct Grid2_ref_xy : public Grid2_ref_slice<T> {

    typedef Grid2_ref_slice<T> Pt;

    Grid2_ref_xy( Grid3_cu<T>& g, Range x, Range y, int z ) :
        Grid2_ref_slice<T>(g, z, x, y)
    {
        init_size( Vec2i_cu(x.nb_elts(), y.nb_elts()), g.get_padd_offset().xy() );
    }

    T& get_val(int x, int y) {
        assert( x < Pt::_size.x && y < Pt::_size.y );
        assert( x >= 0 && y >= 0);
        return Pt::_grid_ref(Pt::_r0._a + x, Pt::_r1._a + y, Pt::_slice);
    }

    const T& get_val(int x, int y) const {
        assert( x < Pt::_size.x && y < Pt::_size.y );
        assert( x >= 0 && y >= 0);
        return Pt::_grid_ref(Pt::_r0._a + x, Pt::_r1._a + y, Pt::_slice);
    }
};

// -----------------------------------------------------------------------------

template<typename T>
struct Grid2_ref_xz : public Grid2_ref_slice<T> {

    typedef Grid2_ref_slice<T> Pt;

    Grid2_ref_xz( Grid3_cu<T>& g, Range x, int y, Range z) :
        Grid2_ref_slice<T>(g, y, x, z)
    {
        init_size( Vec2i_cu(x.nb_elts(), z.nb_elts()), g.get_padd_offset().xz() );
    }

    T& get_val(int x, int y) {
        assert( x < Pt::_size.x && y < Pt::_size.y );
        assert( x >= 0 && y >= 0);
        return Pt::_grid_ref(Pt::_r0._a + x, Pt::_slice, Pt::_r1._a + y);
    }

    const T& get_val(int x, int y) const {
        assert( x < Pt::_size.x && y < Pt::_size.y );
        assert( x >= 0 && y >= 0);
        return Pt::_grid_ref(Pt::_r0._a + x, Pt::_slice, Pt::_r1._a + y);
    }
};

// -----------------------------------------------------------------------------

template<typename T>
struct Grid2_ref_yz : public Grid2_ref_slice<T> {

    typedef Grid2_ref_slice<T> Pt;

    Grid2_ref_yz( Grid3_cu<T>& g, int x , Range y, Range z) :
        Grid2_ref_slice<T>(g, x, y, z)
    {
        init_size( Vec2i_cu(y.nb_elts(), z.nb_elts()), g.get_padd_offset().yz() );
    }

    T& get_val(int x, int y){
        assert( x < Pt::_size.x && y < Pt::_size.y );
        assert( x >= 0 && y >= 0);
        return Pt::_grid_ref(Pt::_slice, Pt::_r0._a + x, Pt::_r1._a + y);
    }
    const T& get_val(int x, int y) const {
        assert( x < Pt::_size.x && y < Pt::_size.y );
        assert( x >= 0 && y >= 0);
        return Pt::_grid_ref(Pt::_slice, Pt::_r0._a + x, Pt::_r1._a + y);
    }
};

// -----------------------------------------------------------------------------
// Implemens of Grid2_ref_const
// -----------------------------------------------------------------------------

template< typename T>
Grid2_const_ref<T>::Grid2_const_ref() : _grid_ref_const(0), _counter(0) {
    _grid_ref_allocated = false;
    _counter  = new Ref_counter();
    _counter->add_ref();
}

// -----------------------------------------------------------------------------

template< typename T>
Grid2_const_ref<T>::Grid2_const_ref( const Grid2_cu<T>& g){
    _grid_ref_allocated = false;
    _grid_ref_const = &g;
    _counter        = new Ref_counter();
    _counter->add_ref();
}

// -----------------------------------------------------------------------------

template< typename T>
Grid2_const_ref<T>

Grid2_const_ref<T>::make_xy(const Grid3_cu<T>& g, Range x, Range y, int z) {
    Grid2_const_ref<T> ref;
    ref._grid_ref_allocated = true;
    // Ok its a hack but it avoids to complexify the architecture
    // '_grid_ref_const' is constant so 'g' should not be modified through it
    // only Grid2_ref_xy constructor might do that but I've make sure it doesn't
    ref._grid_ref_const = new Grid2_ref_xy<T>( const_cast<Grid3_cu<T>&>(g), x, y, z);
    // Memory leak here counter already in default ctor
    //ref._counter        = new Ref_counter();
    //ref._counter->add_ref();
    return ref;
}

// -----------------------------------------------------------------------------

template< typename T>
Grid2_const_ref<T>

Grid2_const_ref<T>::make_xz(const Grid3_cu<T>& g, Range x, int y, Range z) {
    Grid2_const_ref<T> ref;
    ref._grid_ref_allocated = true;
    // see 'make_xy()' for the hack
    ref._grid_ref_const = new Grid2_ref_xz<T>(const_cast<Grid3_cu<T>&>(g), x, y, z);
    // Memory leak here counter already in default ctor
    //ref._counter        = new Ref_counter();
    //ref._counter->add_ref();
    return ref;
}

// -----------------------------------------------------------------------------

template< typename T>
Grid2_const_ref<T>

Grid2_const_ref<T>::make_yz(const Grid3_cu<T>& g, int x, Range y, Range z) {
    Grid2_const_ref<T> ref;
    ref._grid_ref_allocated = true;
    // see 'make_xy()' for the hack
    ref._grid_ref_const = new Grid2_ref_yz<T>(const_cast<Grid3_cu<T>&>(g), x, y, z);
    // Memory leak here counter already in default ctor
    //ref._counter        = new Ref_counter();
    //ref._counter->add_ref();
    return ref;
}

// -----------------------------------------------------------------------------

template< typename T>
Grid2_const_ref<T>::Grid2_const_ref( const Grid2_const_ref<T>& cp ):
    _grid_ref_allocated( cp._grid_ref_allocated ),
    _grid_ref_const( cp._grid_ref_const ),
    _counter( cp._counter )
{
    _counter->add_ref();
}

// -----------------------------------------------------------------------------

template< typename T>
Grid2_const_ref<T>::Grid2_const_ref( const Grid2_ref<T>& cp ):
    _grid_ref_allocated( cp._grid_ref_allocated ),
    _grid_ref_const( cp._grid_ref_const ),
    _counter( cp._counter )
{
    _counter->add_ref();
}

// -----------------------------------------------------------------------------

template< typename T>
Grid2_const_ref<T>::~Grid2_const_ref( )
{
    if(_counter->release() == 0)
    {
        // When false means the pointer has been allocated
        // externally of Grid2_ref
        if( _grid_ref_allocated ) delete _grid_ref_const;
        delete _counter;
    }
}

// -----------------------------------------------------------------------------

template< typename T>
const T& Grid2_const_ref<T>::operator() (int x, int y) const {
    return (*_grid_ref_const)(x, y);
}

// -----------------------------------------------------------------------------

template< typename T>
const T& Grid2_const_ref<T>::operator() (const Idx2_cu& idx) const {
    Vec2i_cu v = idx.to_2d();
    return (*_grid_ref_const)(v.x, v.y);
}

// -----------------------------------------------------------------------------
