/*
 * This software is governed by the CeCILL-B license under French law and
 * abiding by the rules of distribution of free software.  You can  use, 
 * modify and/ or redistribute the software under the terms of the CeCILL-B
 * license as circulated by CEA, CNRS and INRIA at the following URL
 * "http://www.cecill.info" or the LICENCE.txt file present in this project.
*/

#ifndef GRID2_CU_HPP__
#define GRID2_CU_HPP__

#include "grid3_cu.hpp"
#include "grid2_ref.hpp"
#include "vec2i_cu.hpp"
#include "idx2_cu.hpp"

// -----------------------------------------------------------------------------

/** @class Grid2_cu
  * @brief utility to store/use 2d grids
  * @tparam T : Cell type. Default constructor MUST be defined.
  *
  * TODO: doc usage
  *
  * @warning If you inherit from this class follow the rules:
  * children must keep '_size' and '_pad_off' attributes updated,
  * all virtual functions are to be overriden properly
  */
template<typename T>
struct Grid2_cu : public Grid2_ref<T> {

    // implem notes:
    // Warning: don't forget to call Grid2_ref<T>() if you add a new constructor
    // Warning: never use directly the std::vector to copy between grids always
    // use the overloaded accessor ( , ) which can be overriden by a children.

    Grid2_cu(const Vec2i_cu& size_ = Vec2i_cu::zero(),
             const T& val = T(),
             const Vec2i_cu& pad = Vec2i_cu::zero());

    Grid2_cu(const Vec2i_cu& s,
             const T* vals,
             const Vec2i_cu& pad = Vec2i_cu::zero());

    Grid2_cu(const Grid2_cu& cp);


    /// Assign from another reference Grid2_ref -> hard copy
    virtual Grid2_cu<T>& operator=(Grid2_const_ref<T> cp);

    /// When assign another Grid2_cu we do an hard copy
    virtual Grid2_cu<T>& operator= (const Grid2_cu<T>& cp);


    // -------------------------------------------------------------------------
    /// @name Accessors
    // -------------------------------------------------------------------------

    /// Size allocated to store the grid
    Vec2i_cu alloc_size() const { return _size; }

    /// Logical size of the grid using padding offset
    Vec2i_cu size() const { return _size -_pad_off * 2; }

    Vec2i_cu get_padd_offset() const { return _pad_off; }

    const std::vector<T>& get_vals() const { return _vals; }

    /// Access linear storage of the grid where:
    /// linear_idx = x  +  (_size.x) * y
    const T* ptr() const { return &(_vals[0]); }

    // -------------------------------------------------------------------------
    /// @name Access array element
    /// @note Accessors are declined with diferent types (const/non-const etc.)
    /// but their imlementations depends on the virtual one,
    /// so you only need to overide one operator :)
    // -------------------------------------------------------------------------

    virtual       T& get_val(int x, int y);
    virtual const T& get_val(int x, int y) const;

    /// non-const accessors
    /// @{
    Grid2_cu<T>& operator() (Range x, Range y  );
    T&           operator() (int x, int y      );
    T&           operator() (const Idx2_cu& idx);
    /// @}

    /// const accessors
    /// @{
    const Grid2_cu<T>& operator() (Range x, Range y    ) const;
    const T&           operator() (int   x, int   y    ) const;
    const T&           operator() ( const Idx2_cu& idx ) const;
    /// @}

private:
    // implem note: its safer if only this class is allowed to allocate its
    // own memory, children should not be allowed to fiddle with it
    void init_vals(const T* vals);

    std::vector<T> _vals;    ///< Linear storage of the 3D grid
protected:
    Vec2i_cu       _size;
    Vec2i_cu       _pad_off; ///< padding offsets (nb elts)
};

// -----------------------------------------------------------------------------

#include "grid2_cu.inl"

#endif // GRID2_CU_HPP__
