/*
 * This software is governed by the CeCILL-B license under French law and
 * abiding by the rules of distribution of free software.  You can  use, 
 * modify and/ or redistribute the software under the terms of the CeCILL-B
 * license as circulated by CEA, CNRS and INRIA at the following URL
 * "http://www.cecill.info" or the LICENCE.txt file present in this project.
*/

#include "grid2_cu.hpp"

static inline bool check_bounds(const Range& r, int size){
    return r._a >= 0 && r._a < size && (r._b - 1) >= 0 && (r._b - 1) < size;
}

// -----------------------------------------------------------------------------
// Implemens of Grid3_cu to avoid cross definitions
// -----------------------------------------------------------------------------

template< typename T>
Grid2_ref<T>

Grid3_cu<T>::operator() (Range x, Range y, int z) {
    assert( z < size().z );
    if( x._dyn_range ) x._b = size().x;
    if( y._dyn_range ) y._b = size().y;
    assert( check_bounds(x, size().x) );
    assert( check_bounds(y, size().y) );
    return Grid2_ref<T>::make_xy(*this, x, y, z);
}

//------------------------------------------------------------------------------

template< typename T>
Grid2_ref<T>

Grid3_cu<T>::operator() (Range x, int y  , Range z){
    assert( y < size().y );
    if( x._dyn_range ) x._b = size().x;
    if( z._dyn_range ) z._b = size().z;
    assert( check_bounds(x, size().x) );
    assert( check_bounds(z, size().z) );
    return Grid2_ref<T>::make_xz(*this, x, y, z);
}

//------------------------------------------------------------------------------

template< typename T>
Grid2_ref<T>

Grid3_cu<T>::operator() (int x  , Range y, Range z){
    assert( x < size().x );
    if( y._dyn_range ) y._b = size().y;
    if( z._dyn_range ) z._b = size().z;
    assert( check_bounds(y, size().y) );
    assert( check_bounds(z, size().z) );
    return Grid2_ref<T>::make_yz(*this, x, y, z);
}

//------------------------------------------------------------------------------

template< typename T>
const Grid2_const_ref<T>

Grid3_cu<T>::operator() (Range x, Range y, int z) const {
    assert( z < size().z );
    if( x._dyn_range ) x._b = size().x;
    if( y._dyn_range ) y._b = size().y;
    assert( check_bounds(x, size().x) );
    assert( check_bounds(y, size().y) );
    return Grid2_ref<T>::make_xy(*this, x, y, z);
}

//------------------------------------------------------------------------------

template< typename T>
const Grid2_const_ref<T>

Grid3_cu<T>::operator() (Range x, int y  , Range z) const {
    assert( y < size().y );
    if( x._dyn_range ) x._b = size().x;
    if( z._dyn_range ) z._b = size().z;
    assert( check_bounds(x, size().x) );
    assert( check_bounds(z, size().z) );
    return Grid2_ref<T>::make_xz(*this, x, y, z);
}

//------------------------------------------------------------------------------

template< typename T>
const Grid2_const_ref<T>

Grid3_cu<T>::operator() (int x  , Range y, Range z) const {
    assert( x < size().x );
    if( y._dyn_range ) y._b = size().y;
    if( z._dyn_range ) z._b = size().z;
    assert( check_bounds(y, size().y) );
    assert( check_bounds(z, size().z) );
    return Grid2_ref<T>::make_yz(*this, x, y, z);
}

// -----------------------------------------------------------------------------
// Implemens of Grid2_cu to avoid cross definitions
// -----------------------------------------------------------------------------

template< typename T>
Grid2_cu<T>::Grid2_cu(const Vec2i_cu& size_,
                      const T& val,
                      const Vec2i_cu& pad) :
    Grid2_ref<T>( *this ),
    _vals( size_.product(), val),
    _size( size_ ),
    _pad_off( pad )
{

}

// -----------------------------------------------------------------------------

template< typename T>
Grid2_cu<T>::Grid2_cu(const Vec2i_cu& s,
                      const T* vals,
                      const Vec2i_cu& pad) :
    Grid2_ref<T>( *this ),
    _size( s ),
    _pad_off( pad )
{
    init_vals( vals );
}

// -----------------------------------------------------------------------------

template< typename T>
Grid2_cu<T>::Grid2_cu(const Grid2_cu& cp) :
    Grid2_ref<T>( *this ),
    _size( cp._size ),
    _pad_off( cp._pad_off ),
    _vals( _size.product() )
{
    for( Idx2_cu idx(_size, 0); idx.is_in(); ++idx )
        _vals[idx.to_linear()] = cp( idx );
}

// -----------------------------------------------------------------------------

template< typename T>
Grid2_cu<T>& Grid2_cu<T>::operator=(Grid2_const_ref<T> cp)
{
    _size    = cp.alloc_size();
    _pad_off = cp.get_padd_offset();
    _vals.resize( _size.product() );
    for( Idx2_cu idx(_size, 0); idx.is_in(); ++idx )
        _vals[idx.to_linear()] = cp( idx );

    return *this;
}

// -----------------------------------------------------------------------------

template< typename T>
Grid2_cu<T>& Grid2_cu<T>::operator= (const Grid2_cu<T>& cp)
{
    _size    = cp._size;
    _pad_off = cp._pad_off;
    _vals.resize( _size.product() );
    for( Idx2_cu idx(_size, 0); idx.is_in(); ++idx )
        _vals[idx.to_linear()] = cp( idx );
    return *this;
}

// -----------------------------------------------------------------------------

// Implem note:
// non const 2D accessor accessors must use this one so that everyone
// behave the same when get_val is overriden
template< typename T>
T& Grid2_cu<T>::get_val(int x, int y)
{
    Idx2_cu idx = Idx2_cu(_size, _pad_off) + Vec2i_cu(x, y);
    return _vals[idx.to_linear()];
}

// -----------------------------------------------------------------------------

// Implem note:
// const 2D accessor accessors must use this one so that everyone
// behave the same when get_val is overriden
template< typename T>
const T& Grid2_cu<T>::get_val(int x, int y) const
{
    Idx2_cu idx = Idx2_cu(_size, _pad_off) + Vec2i_cu(x, y);
    return _vals[idx.to_linear()];
}

// -----------------------------------------------------------------------------

template< typename T>
T& Grid2_cu<T>::operator() (int x, int y) { return get_val(x, y); }

// -----------------------------------------------------------------------------

template< typename T>
T& Grid2_cu<T>::operator() ( const Idx2_cu& idx )
{
    Vec2i_cu v = idx.to_2d();
    // We use the virtual operator as it might be overidden.
    return get_val(v.x, v.y);
}

// -----------------------------------------------------------------------------

template< typename T>
Grid2_cu<T>& Grid2_cu<T>::operator() (Range, Range) { return (*this); }

// -----------------------------------------------------------------------------

template< typename T>
const T& Grid2_cu<T>::operator() (int x, int y) const { return get_val(x, y); }

// -----------------------------------------------------------------------------

template< typename T>
const T& Grid2_cu<T>::operator() ( const Idx2_cu& idx ) const {
    Vec2i_cu v = idx.to_2d();
    // We use the virtual operator as it might be overidden.
    return get_val(v.x, v.y);
}

// -----------------------------------------------------------------------------

template< typename T>
const Grid2_cu<T>& Grid2_cu<T>::operator() (Range, Range) const {
    return (*this);
}

// -----------------------------------------------------------------------------

template< typename T>
void Grid2_cu<T>::init_vals(const T* vals)
{
    int nb_elt = _size.product();
    _vals.resize( nb_elt );
    for (int i = 0; i < nb_elt; ++i)
        _vals[i] = vals[i];
}

// -----------------------------------------------------------------------------
