/*
 * This software is governed by the CeCILL-B license under French law and
 * abiding by the rules of distribution of free software.  You can  use, 
 * modify and/ or redistribute the software under the terms of the CeCILL-B
 * license as circulated by CEA, CNRS and INRIA at the following URL
 * "http://www.cecill.info" or the LICENCE.txt file present in this project.
*/

#ifndef GRID2_REF_HPP__
#define GRID2_REF_HPP__

#include "grid2_const_ref.hpp"

/**
 * @name Grid2_ref
 * @brief Class encapsulating a pointer of a Grid2_cu (like a smart pointer)
 *
 * This smart pointer like class for 2D grids should be seen as a reference
 * to a polymorphic 2D grid Grid2_cu. Unlike smart pointers we don't overload
 * '*' or '->'. Instead we overload the usuals accesors of a Grid2_cu
 * (for instance access a grid element with operator (x,y) ).
 * Therefore the class is more like a "smart reference".
 *
 * You can build a reference from any 2D grid that inherit from #Grid2_cu
 * @code
 * // Below passing a Grid2_cu argument or a children as parameter of 'fun()'
 * // will only copy pointers
 * template<typename T>
 * void fun( Grid2_ref<T> g_ref ){ g_ref(x,y) = value; }
 *
 * {
 *     Grid2_cu <T> grid2;
 *     Grid2_ref<T> a_ref( grid2 );
 *     fun( grid2 ); // No extra copy of grid2
 *     fun( a_ref ); // Do the same as the previous line
 * }
 * @endcode
 *
 * But what's the point of the class, we could just use 'Grid2_cu&'?
 * Actually the class hide allocation/deletion of memory in this case:
 * @code
 * // Example of a standard way to handle class polymorphism
 * template<typename T>
 * void foobar( Grid2_cu<T>& g ){ g(x,y) = value; }
 *
 * {
 *    Grid2_ref_xy<T>* g_ptr = new Grid2_ref_xy<T>( ... );
 *    foobar( *g_ptr ); // specific behavior
 *    delete g_ptr;
 * }
 *
 * // Example with Grid2_ref which doesn't need memory handling
 * template<typename T>
 * void foobar2( Grid2_ref<T> g ){ g(x,y) = value; }
 *
 * {
 *    // This constructor internally do a 'new Grid2_ref_xy<T>( ... );'
 *    Grid2_ref<T> g_ref( grid3, 5);
 *    foobar( g_ptr ); // specific behavior
 *    // deletion is done automatically in Grid2_ref destructor
 * }
 * @endcode
 *
 * If you need a "constant smart reference" use Grid2_const_ref which is
 * the same as Grid2_ref but guatantees to not change what is referenced.
 * Alternatively you can also use a 'const Grid2_ref&' which is equivalent to
 * a 'Grid2_const_ref'.
 *
 * @warning: child inheriting from this class should not forget to call
 * Grid2_ref<T>() constructor. Otherwise you are likely to segfault if a child
 * is passed as an argument of type Grid2_ref/Grid2_const_ref
 *
 * @warning: Prefer to pass directly by value Grid2_ref/Grid2_const_ref.
 * You can't use types like 'Grid2_ref&' or 'Grid2_const_ref&'. First it's
 * uselless as the copy constructor of these are very lightweight. Second it
 * will prevent to use many functions returning  Grid2_ref/Grid2_const_ref by
 * value (in C++ you cannot assign to a non-const reference a temporary value).
 * You could use 'const Grid2_const_ref&' which is also equivalent to
 * 'const Grid2_ref&'.
 *
 */
template<typename T>
struct Grid2_ref : public Grid2_const_ref<T> {

    Grid2_ref( Grid2_cu<T>& g);

    static Grid2_ref<T> make_xy(Grid3_cu<T>& g, Range x, Range y, int   z);
    static Grid2_ref<T> make_xz(Grid3_cu<T>& g, Range x, int   y, Range z);
    static Grid2_ref<T> make_yz(Grid3_cu<T>& g, int   x, Range y, Range z);

    Grid2_ref( const Grid2_ref<T>& cp );

    /// Only destroy if never referenced in another shallow copy of Grid2_ref
    virtual ~Grid2_ref( ){ }

    /// Assignement is an hard copy (copy every values of the grid as we would
    /// expect from a Grid2_cu& reference)
    Grid2_cu<T>& operator= (Grid2_const_ref<T> cp);
    Grid2_cu<T>& operator= (Grid2_ref<T>       cp);

    // -------------------------------------------------------------------------
    // Access referenced data:
    // -------------------------------------------------------------------------

    /// @defgroup Non const accessors
    /// @{
    T& operator() (int x, int y);
    T& operator() (const Idx2_cu& idx);
    /// @}

    /// @defgroup Const accessors
    // Even though we should inherit them from Grid2_const_ref we have to
    // re-define them as they get hidden by the non-const version. Yes its
    // another C++ pitfall...
    /// @{
    const T& operator() (int x, int y)       const;
    const T& operator() (const Idx2_cu& idx) const;
    /// @}

private:
    /// Don't make sense to have a default constructor for a reference
    /// However we need it internally
    Grid2_ref();

    /// Factor code when assign to another reference const and non const
    void copy_from( const Grid2_const_ref<T>& cp );

    /// Polymorphic pointer to any 2D grid
    Grid2_cu<T>* _grid_ref;
};


#include "grid2_ref.inl"

#endif // GRID2_REF_HPP__
