/*
 * This software is governed by the CeCILL-B license under French law and
 * abiding by the rules of distribution of free software.  You can  use, 
 * modify and/ or redistribute the software under the terms of the CeCILL-B
 * license as circulated by CEA, CNRS and INRIA at the following URL
 * "http://www.cecill.info" or the LICENCE.txt file present in this project.
*/

#ifndef GRID3_CU_HPP
#define GRID3_CU_HPP

#include <vector>
#include "idx3_cu.hpp"
#include "vec3i_cu.hpp"
#include "cuda_compiler_interop.hpp"
#include "grid_enums.hpp"

// -----------------------------------------------------------------------------

template< typename T>
struct Grid2_ref;

template< typename T>
struct Grid2_const_ref;

// -----------------------------------------------------------------------------

/** @class Grid3_cu
  * @brief utility to store/use 3d grids with cuda
  * @tparam T : Cell type. Default constructor MUST be defined.
  *
  * TODO: doc usage
  *
  */

//FIXME: handle bool correctly has std::vector won't ...
template<class T>
struct Grid3_cu {

    /// @brief Padding kind enumerant.
    enum Pad_t {
        COPY,   ///< padding will extend border values
        CUSTOM  ///< padding will initialize to specified value
    };

    // -------------------------------------------------------------------------
    /// @name Constructors
    // -------------------------------------------------------------------------

    Grid3_cu(const Vec3i_cu& size = Vec3i_cu::zero(),
             const T& val = T(),
             const Vec3i_cu& pad = Vec3i_cu::zero() );

    Grid3_cu(int x, int y, int z,
             const T* vals,
             const Vec3i_cu& pad = Vec3i_cu::zero());

    Grid3_cu(const Vec3i_cu& s,
             const T* vals,
             const Vec3i_cu& pad = Vec3i_cu::zero());

    /// Copy constructor do a hard copy
    Grid3_cu( const Grid3_cu<T>& g );

    /// Concatenate a list of grids, dimensions are computed according to the
    /// maximal dimension allowed 'max' for the final grid
    /// @param list : list of 3d grids to be concatenated. All grids MUST be
    /// equal in size.
    /// @param max : Maximum size of the new grid resulting from
    /// the concatenation of the grid list.
    /// @param out_idx : returned indices in the new grid to access first
    /// non-padded data for each input grid.
    /// The new grid is filled if necessary but not padded.
    Grid3_cu(const std::vector< Grid3_cu<T>* >& list,
             const Vec3i_cu& max,
             std::vector<Idx3_cu>& out_idx);

    /// Symetric padding of the grid.
    /// @param padding : the number of faces added to each side of the grid.
    /// @param type : How do we fill the new faces? (user defined value,
    /// repeat values by extruding each orinal face etc.)
    /// @param val : padding value when if type == CUSTOM. Otherwise parameter
    /// is ignored
    void padd(int padding, Pad_t type = COPY, T val = T() );


    // -------------------------------------------------------------------------
    /// @name Accessors
    // -------------------------------------------------------------------------

    /// Size allocated to store the grid
    Vec3i_cu alloc_size() const { return _size; }

    /// Logical size of the grid using padding offset
    Vec3i_cu size() const { return _size - _pad_off * 2; }

    Vec3i_cu get_padd_offset() const { return _pad_off; }

    const std::vector<T>& get_vals() const { return _vals; }

    /// Access linear storage of the grid where:
    /// linear_idx = x  +  (_size.x) * y  +  (_size.x * _size.y) * z
    const T* ptr() const { return &(_vals[0]); }

    /// Allocate and copy grid to GPU for future use with cuda textures
    /// @return pointer to newly allocated device mem.
    cudaArray* to_gpu() const;

    /// Assignement op do a hard copy
    Grid3_cu<T>& operator= (const Grid3_cu<T>& cp);


    T& operator() (int x, int y, int z);
    T& operator() (const Idx3_cu& idx);

    const T& operator() (int x, int y, int z) const;
    const T& operator() (const Idx3_cu& idx) const;


    /// @defgroup Acces a 2D xy slice
    // Implemented in grid2_cu.inl because of cross definitions
    /// @{
    Grid2_ref<T> operator() (Range x, Range y, int   z);
    Grid2_ref<T> operator() (Range x, int   y, Range z);
    Grid2_ref<T> operator() (int   x, Range y, Range z);

    const Grid2_const_ref<T> operator() (Range x, Range y, int   z) const;
    const Grid2_const_ref<T> operator() (Range x, int   y, Range z) const;
    const Grid2_const_ref<T> operator() (int   x, Range y, Range z) const;
    /// @}


private:
    void init_vals(const T* vals);

    Vec3i_cu _size;        ///< 3d size of the grid (nb elts)
    Vec3i_cu _pad_off;     ///< padding offsets (nb elts)
    std::vector<T> _vals;  ///< Linear storage of the 3D grid
};

#include "grid3_cu.inl"

#endif // GRID3_CU_HPP
