/*
 * This software is governed by the CeCILL-B license under French law and
 * abiding by the rules of distribution of free software.  You can  use, 
 * modify and/ or redistribute the software under the terms of the CeCILL-B
 * license as circulated by CEA, CNRS and INRIA at the following URL
 * "http://www.cecill.info" or the LICENCE.txt file present in this project.
*/

#ifndef GRID_UTILS_HPP__
#define GRID_UTILS_HPP__

#include <vector>

#include "grid3_cu.hpp"
#include "grid2_cu.hpp"

// =============================================================================
namespace Grid_utils {
// =============================================================================


/// Compute 2d gradients with central finite difference.
/// Borders of the grid are untouched.
/// @param scale : if 'vals' represent a 2D function what is the range of (x, y)
/// variables? Scale defines that x ranges to [0; scale.x] and y ranges to
/// [0; scale.y].
/// @tparam Real  : a real number type
/// @tparam Grad2 : gradient must define '.x' and '.y' attributes
template<typename Real, typename Grad2>
void grad_2d(Grid2_ref<Real> vals,
             Grid2_ref<Grad2> grads,
             const Vec2_cu scale = Vec2_cu(1., 1.))
{
    assert( vals.size() == grads.size());

    Real step_x = scale.x / (vals.size().x - 1);
    Real step_y = scale.y / (vals.size().y - 1);

    Idx2_cu off( vals.size(), 1, 1);
    for(Idx2_cu sub_idx( vals.size()-2, 0); sub_idx.is_in(); ++sub_idx)
    {
        Idx2_cu idx = off + sub_idx.to_2d();

        Real x_plus_h  = vals( idx + Vec2i_cu( 1, 0) );
        Real x_minus_h = vals( idx + Vec2i_cu(-1, 0) );
        Real grad_x = (x_plus_h - x_minus_h) / (2. * step_x);

        Real y_plus_h  = vals( idx + Vec2i_cu(0,  1) );
        Real y_minus_h = vals( idx + Vec2i_cu(0, -1) );
        Real grad_y = (y_plus_h - y_minus_h) / (2. * step_y);

        grads( idx ).x = grad_x;
        grads( idx ).y = grad_y;
    }
}

// -----------------------------------------------------------------------------

/// @param grid : grid of values/vectors you want to diffuse
/// @param mask : values to '0' are not diffused
/// @tparam T : must implement constructor T( double ) and overload
/// 'T operator/(double)' 'T operator+(T)'
template< typename T>
void diffuse(Grid2_ref<T> grid, Grid2_const_ref<int> mask, int nb_iter = 128*16)
{
    assert( grid.size() == mask.size());
    Vec2i_cu neighs[4] = {Vec2i_cu( 1, 0),
                          Vec2i_cu( 0, 1),
                          Vec2i_cu(-1, 0),
                          Vec2i_cu( 0,-1)};

    // First we store every grid elements that need to be diffused according
    // to 'mask'
    int mask_nb_elts = mask.size().product();
    std::vector<int> idx_list;
    idx_list.reserve( mask_nb_elts );
    for(Idx2_cu idx(mask.size(), 0); idx.is_in(); ++idx) {
        if( mask( idx ) == 1 )
            idx_list.push_back( idx.to_linear() );
    }

    // Diffuse values with a Gauss-seidel like scheme
    for(int j = 0; j < nb_iter; ++j)
    {
        // Look up grid elements to be diffused
        for(unsigned i = 0; i < idx_list.size(); ++i)
        {
            Idx2_cu idx(grid.size(), idx_list[i]);

            T sum(0.);
            int nb_neigh = 0;

            for(int i = 0; i < 4; ++i)
            {
                Idx2_cu idx_neigh = idx + neighs[i];
                if( idx_neigh.is_in() /*check if inside grid*/ ){
                    sum = sum + grid( idx_neigh );
                    nb_neigh++;
                }
            }

            sum = sum / (double)( nb_neigh );
            grid( idx ) = sum;
        }
    }
}

// -----------------------------------------------------------------------------

#if 0
// Test not sure its correct
template< typename T>
void diffuse_biharmonic(Grid2_ref<T> grid,
                        Grid2_const_ref<int> mask,
                        int nb_iter = 128*16)
{
    assert( grid.size() == mask.size());
    Vec2i_cu neighs[12] = {Vec2i_cu( 1, 0),
                          Vec2i_cu( 0, 1),
                          Vec2i_cu(-1, 0),
                          Vec2i_cu( 0,-1),

                          Vec2i_cu( 1, 1),
                          Vec2i_cu(-1,-1),
                          Vec2i_cu(-1, 1),
                          Vec2i_cu( 1,-1),

                          Vec2i_cu( 2, 0),
                          Vec2i_cu( 0, 2),
                          Vec2i_cu(-2, 0),
                          Vec2i_cu( 0,-2) };

    // First we store every grid elements that need to be diffused according
    // to 'mask'
    int mask_nb_elts = mask.size().product();
    std::vector<int> idx_list;
    idx_list.reserve( mask_nb_elts );
    for(Idx2_cu idx(mask.size(), 0); idx.is_in(); ++idx) {
        if( mask( idx ) == 1 )
            idx_list.push_back( idx.to_linear() );
    }

    // Diffuse values with a Gauss-seidel like scheme
    for(int j = 0; j < nb_iter; ++j)
    {
        // Look up grid elements to be diffused
        for(unsigned i = 0; i < idx_list.size(); ++i)
        {
            Idx2_cu idx(grid.size(), idx_list[i]);

            T sum(0.);
            int nb_neigh = 0;

            for(int i = 0; i < 12; ++i)
            {
                Idx2_cu idx_neigh = idx + neighs[i];
                if( idx_neigh.is_in() /*check if inside grid*/ ){
                    sum = sum + grid( idx_neigh );
                    nb_neigh++;
                }
            }

            sum = sum / T( nb_neigh );
            grid( idx ) = sum;
        }
    }
}
#endif

// -----------------------------------------------------------------------------

template< typename T>
void mirror_anti_diag_lower( Grid2_ref<T> grid )
{
    for (int j = 0; j < grid.size().y; ++j) {
        for (int i = j+1; i < grid.size().x; ++i) {
            grid(j, i) = grid(i, j);
        }
    }
}

// -----------------------------------------------------------------------------

/// @note Diagonale goes from upper left to lower right and anti-diagonale
/// goes from lower left to upper right.
enum Mirror_diag_t {
    LOWER,      ///< Mirror strict lower triangle (formed by the diagonal)
    UPPER,      ///< Mirror strict upper triangle (formed by the diagonal)
    ANTI_UPPER, ///< Mirror strict upper triangle (formed by the anti-diagonal)
    ANTI_LOWER  ///< Mirror strict lower triangle (formed by the anti-diagonal)
};

template< typename T>
void mirror_diag(Grid2_ref<T> grid, Mirror_diag_t type)
{
    // Must be a square grid
    assert( grid.size().x == grid.size().y);
    switch(type){
    case ANTI_LOWER: mirror_anti_diag_lower( grid ); break;
    default:
        assert(false); // Not implemented
    }
}



}// END NAMESPACE Grid_utils ===================================================


#endif // GRID_UTILS_HPP__
