/*
 * This software is governed by the CeCILL-B license under French law and
 * abiding by the rules of distribution of free software.  You can  use, 
 * modify and/ or redistribute the software under the terms of the CeCILL-B
 * license as circulated by CEA, CNRS and INRIA at the following URL
 * "http://www.cecill.info" or the LICENCE.txt file present in this project.
*/

#ifndef IDX2_CU_HPP
#define IDX2_CU_HPP

#include "cuda_compiler_interop.hpp"
#include "vec2i_cu.hpp"

/**
 * @struct Idx2_cu
 * @brief 2d grid index with conversion to 2d and linear storage
 *
 * Use case to look up every elements of a grid of 16x16
 * @code
 *
 * Vec2i_cu s(16, 16);
 * for(Idx2_cu idx(s, 0); idx.is_in(); ++idx)
 * {
 *      Vec2i_cu idx_2d = idx.to_2d(); // or .to_vec2i() will do the same
 *      int      i      = idx.to_linear();
 *
 *      // i == idx_2d.x + s.x * idx_2d.y
 * }
 * @endcode
 *
 * Looking up a sub grid 10^2 inside a grid 128^2:
 * @code
 * Vec2i_cu sub_grid_s(10, 10);
 * Vec2i_cu grid_s(128, 128);
 * Idx2_cu offset(grid_s, offx, offy);
 * for(Idx2_cu idx(sub_grid_s, 0); idx.is_in(); ++idx)
 * {
 *     int i = (offset + idx.to_vec2i()).to_linear();
 *     // 'i' is in linear coordinates of the grid 128^2 but will only look up
 *     // a sub block of size 10^2 starting from the offset in the 128^2 grid
 * }
 * @endcode
 */

struct Idx2_cu {


    // -------------------------------------------------------------------------
    /// @name Constructors
    // -------------------------------------------------------------------------

    IF_CUDA_DEVICE_HOST inline
    Idx2_cu() : _size(-1, -1), _id(-1) { }

    /// Build index from a linear index
    IF_CUDA_DEVICE_HOST inline
    Idx2_cu(const Vec2i_cu& size, int idx) : _size(size), _id(idx) { }

    /// Build index from a 3d index
    IF_CUDA_DEVICE_HOST inline
    Idx2_cu(const Vec2i_cu& size, int ix, int iy) : _size(size) {
        _id = to_linear(_size, ix, iy);
    }

    /// Build index from a 3d index
    IF_CUDA_DEVICE_HOST inline
    Idx2_cu(const Vec2i_cu& size, const Vec2i_cu& pos) : _size(size) {
        set_2d( pos );
    }

    // -------------------------------------------------------------------------
    /// @name Set index position
    // -------------------------------------------------------------------------

    IF_CUDA_DEVICE_HOST inline
    void set_linear(int i){ _id = i; }

    IF_CUDA_DEVICE_HOST inline
    void set_2d(const Vec2i_cu& p){ set_2d(p.x, p.y); }

    IF_CUDA_DEVICE_HOST inline
    void set_2d(int x, int y){ _id = to_linear(_size, x, y); }

    IF_CUDA_DEVICE_HOST inline
    int to_linear() const { return _id; }

    // -------------------------------------------------------------------------
    /// @name Get index position
    // -------------------------------------------------------------------------

    IF_CUDA_DEVICE_HOST inline
    Vec2i_cu to_2d() const { Vec2i_cu r; to_2d(r.x, r.y); return r; }

    IF_CUDA_DEVICE_HOST inline
    Vec2i_cu to_vec2i() const { return to_2d(); }

    IF_CUDA_DEVICE_HOST inline
    void to_2d(int& x, int& y) const {
        x = _id % _size.x;
        y = _id / _size.x;
    }

    // -------------------------------------------------------------------------
    /// @name Other methods
    // -------------------------------------------------------------------------

#ifdef __CUDACC__
    int3 to_int3() const { return make_int3(_size.x, _size.y, _id); }
#endif

    IF_CUDA_DEVICE_HOST inline
    int size_linear() const { return _size.product(); }

    IF_CUDA_DEVICE_HOST inline
    Vec2i_cu size() const { return _size; }

    /// A valid index is positive as well as its size
    IF_CUDA_DEVICE_HOST inline
    bool is_valid() const {
        return _id >= 0 && size_linear() >= 0;
    }

    /// Does the index is out of its bounds (defined at construction)
    IF_CUDA_DEVICE_HOST inline bool is_out() const { return !is_in(); }

    /// Does the index is inside its bounds (defined at construction)
    IF_CUDA_DEVICE_HOST inline
    bool is_in() const {
        return (_id < size_linear()) && (_id >= 0);
    }

    // -------------------------------------------------------------------------
    /// @name Operators overload
    // -------------------------------------------------------------------------

    IF_CUDA_DEVICE_HOST inline
    Idx2_cu operator++(   ) { return Idx2_cu(_size, ++_id); }

    IF_CUDA_DEVICE_HOST inline Idx2_cu operator++(int)
    { return Idx2_cu(_size, _id++); }

    IF_CUDA_DEVICE_HOST inline
    Idx2_cu operator--(   ) { return Idx2_cu(_size, --_id); }

    IF_CUDA_DEVICE_HOST inline
    Idx2_cu operator--(int) { return Idx2_cu(_size, _id--); }

    IF_CUDA_DEVICE_HOST inline
    bool operator==(const Idx2_cu& i) const {
        return _size == i._size && _id == i._id;
    }

    IF_CUDA_DEVICE_HOST inline
    bool operator!=(const Idx2_cu& i) const {
        return _size != i._size || _id != i._id;
    }

    IF_CUDA_DEVICE_HOST inline
    Idx2_cu operator =(const Idx2_cu& i) {
        _size = i._size; _id = i._id; return *this;
    }

    IF_CUDA_DEVICE_HOST inline friend
    Idx2_cu operator+ (const Idx2_cu& id, const Vec2i_cu& v) {
        Vec2i_cu this_idx = id.to_2d();
        return Idx2_cu(id._size, this_idx + v);
    }

    IF_CUDA_DEVICE_HOST inline friend
    Idx2_cu operator+ (const Vec2i_cu& v, const Idx2_cu& id) {
        return id + v;
    }

private:

    IF_CUDA_DEVICE_HOST static inline
    int to_linear(const Vec2i_cu& size, int x, int y) {
        return x + size.x * y;
    }

    Vec2i_cu _size; ///< 3d size the index is looking up
    int      _id;   ///< Linear index

    // WARNING: these operators should not be used/implemented since:
    // (they don't really make sense) || (are to ambigus to decypher when used)
#if 0
    bool operator<=(const Idx2_cu& ) const { return false; }
    bool operator>=(const Idx2_cu& ) const { return false; }
    bool operator< (const Idx2_cu& ) const { return false; }
    bool operator> (const Idx2_cu& ) const { return false; }

    Idx2_cu operator- (const Idx2_cu& ) const { return Idx2_cu(); }
    Idx2_cu operator+ (const Idx2_cu& ) const { return Idx2_cu(); }
    Idx2_cu operator+=(const Idx2_cu& )       { return Idx2_cu(); }
    Idx2_cu operator-=(const Idx2_cu& )       { return Idx2_cu(); }

    bool operator==(int ) const { return false; }
    bool operator!=(int ) const { return false; }
    bool operator<=(int ) const { return false; }
    bool operator>=(int ) const { return false; }
    bool operator> (int ) const { return false; }
    bool operator< (int ) const { return false; }

    Idx2_cu operator+ (int )  const { return Idx2_cu(); }
    Idx2_cu operator- (int )  const { return Idx2_cu(); }
    Idx2_cu operator+=(int )        { return Idx2_cu(); }
    Idx2_cu operator-=(int )        { return Idx2_cu(); }
#endif
};

#endif // IDX2_CU_HPP
