/*
 * This software is governed by the CeCILL-B license under French law and
 * abiding by the rules of distribution of free software.  You can  use, 
 * modify and/ or redistribute the software under the terms of the CeCILL-B
 * license as circulated by CEA, CNRS and INRIA at the following URL
 * "http://www.cecill.info" or the LICENCE.txt file present in this project.
*/

#ifndef DEVICE_ELT_HPP__
#define DEVICE_ELT_HPP__

#include "cuda_utils_common.hpp"
#include <vector>
#include <iostream>

// =============================================================================
namespace Cuda_utils{
// =============================================================================

// =============================================================================
namespace Device{
// =============================================================================

/**
 * @class Elt
 * @brief Automatic upload of an element of type 'T' to GPU
 *
 * Usefull if T is too big to be a kernel parameter.
 *
 * usage:
 * @code
 * __global__
 * void kernell( Data_t* d_data )
 * {
 *     d_data->method();
 * }
 *
 * {
 *      Data_t my_data;
 *      Cuda_utils::Device::Elt<Data_t> d_data( my_data );
 *      kernell<<<xx,xx>>>( d_data.ptr() );
 * } // Automatic deletion of device mem
 *
 * @endcode
 */
template <typename T>
struct Elt {

    Elt(const T& elt) {
        malloc_d( _d_ptr, 1);
        mem_cpy_htd(_d_ptr, &elt, 1);
    }

    ~Elt() { free_d(_d_ptr); }

    /// Pointer to device memory
    T* d_ptr(){ return _d_ptr; }

private:
    T* _d_ptr;
};

}// END Device =================================================================

}// END Cuda_utils =============================================================

#endif //DEVICE_ELT_HPP__
