/*
 * This software is governed by the CeCILL-B license under French law and
 * abiding by the rules of distribution of free software.  You can  use, 
 * modify and/ or redistribute the software under the terms of the CeCILL-B
 * license as circulated by CEA, CNRS and INRIA at the following URL
 * "http://www.cecill.info" or the LICENCE.txt file present in this project.
*/

#include "glbuffer_object.hpp"

// -----------------------------------------------------------------------------

GlBuffer_obj::GlBuffer_obj(int type) :
    _is_cuda_registered(false),
    _size_buffer(0),
    _type(type)

{
    #ifndef NCUDA
        _cuda_rsrc = 0;
    #endif
    _data_ratio = guess_data_ratio(_type);
    glAssert( glGenBuffers(1, &_buffer_id) );
}

// -----------------------------------------------------------------------------

GlBuffer_obj::GlBuffer_obj(int nb_elt, int type, GLenum mode):
    _is_cuda_registered(false),
    _size_buffer(nb_elt),
    _type(type)
{
    #ifndef NCUDA
        _cuda_rsrc = 0;
    #endif

    _data_ratio = guess_data_ratio(type);
    glAssert( glGenBuffers(1, &_buffer_id) );
    glAssert( glBindBuffer(_type, _buffer_id) );
    glAssert( glBufferData(_type, nb_elt * _data_ratio, NULL, mode) );
    glAssert( glBindBuffer(_type, 0) );
}

// -----------------------------------------------------------------------------

GlBuffer_obj::~GlBuffer_obj() {
    assert(!_is_cuda_registered);
    glAssert( glDeleteBuffers(1, &_buffer_id) );
}

// -----------------------------------------------------------------------------

void GlBuffer_obj::bind() const {
    glAssert( glBindBuffer(_type, _buffer_id) );
}

// -----------------------------------------------------------------------------

void GlBuffer_obj::unbind() const {
    glAssert( glBindBuffer(_type, 0) );
}

// -----------------------------------------------------------------------------

void GlBuffer_obj::set_data(int nb_elt,
                            const GLvoid* data,
                            GLenum mode)
{
    bind();
    _size_buffer = nb_elt;
    glAssert( glBufferData(_type, nb_elt * _data_ratio, data, mode) );
    unbind();
}

// -----------------------------------------------------------------------------

void GlBuffer_obj::get_data(int offset,
                            int nb_elt,
                            GLvoid* data) const
{
    bind();
    glAssert( glGetBufferSubData(_type, offset, nb_elt * _data_ratio, data) );
    unbind();
}

// -----------------------------------------------------------------------------

bool GlBuffer_obj::unmap() const {
    bind();
    bool state = glUnmapBuffer(_type) ? true : false;
    GL_CHECK_ERRORS();
    return state;
}

// -----------------------------------------------------------------------------

#ifndef NCUDA
void GlBuffer_obj::cuda_register(unsigned flag_read_write){
    assert(!_is_cuda_registered);
    assert(_size_buffer > 0);

    _is_cuda_registered = true;
    CUDA_SAFE_CALL( cudaGraphicsGLRegisterBuffer(&_cuda_rsrc, _buffer_id, flag_read_write) );
}

// -----------------------------------------------------------------------------

void GlBuffer_obj::cuda_unregister(){
    assert(_is_cuda_registered);
    assert(_size_buffer > 0);
    _is_cuda_registered = false;
    CUDA_SAFE_CALL( cudaGraphicsUnregisterResource( _cuda_rsrc ) );
}

// -----------------------------------------------------------------------------

void GlBuffer_obj::cuda_unmap() {
    assert(_is_cuda_registered);
    assert(_size_buffer > 0);
    CUDA_SAFE_CALL( cudaGraphicsUnmapResources(1, &_cuda_rsrc) );
}
#endif // END CUDA specific code
