/*
 * This software is governed by the CeCILL-B license under French law and
 * abiding by the rules of distribution of free software.  You can  use, 
 * modify and/ or redistribute the software under the terms of the CeCILL-B
 * license as circulated by CEA, CNRS and INRIA at the following URL
 * "http://www.cecill.info" or the LICENCE.txt file present in this project.
*/

#include "glpick_fbo.hpp"

#include "glfbo.hpp"
#include "gltex2D.hpp"
#include "shader.hpp"

#include <cassert>
#include <limits>
#include <vector>

// =============================================================================
namespace Shader_picking {
// =============================================================================

Shader_prog* g_shader = 0;

bool is_shaders_init = false;

int nb_inst = 0;

// -----------------------------------------------------------------------------

// Hard coded phong vertex shader
const char* src_vert =
        "uniform mat4 MVP;\n"
        "void main(){\n"
        "   gl_Position = ftransform(); //MVP * vec4(Position, 1.0);\n"
        "}\n\0";

// -----------------------------------------------------------------------------

// Hard coded phong fragment shader
const char* src_frag =
        "#version 410\n"
        "#extension GL_EXT_gpu_shader4 : enable\n"
        "uniform int g_object_idx;\n"
        "out ivec3 FragColor;\n"
        "void main(){\n"
        "// Plus '1' to not mix up the first elemetn with the background color '0'\n"
        "   FragColor = ivec3(g_object_idx + 1, gl_PrimitiveID + 1, 0);\n"
        "}\n\0";

// -----------------------------------------------------------------------------

void delete_shaders()
{
    delete g_shader;
    g_shader = 0;

    is_shaders_init = false;
}

// -----------------------------------------------------------------------------

void init_shaders()
{
    delete_shaders();

    Shader vs(GL_VERTEX_SHADER  );
    Shader fs(GL_FRAGMENT_SHADER);

    if( !vs.load_source( src_vert ) ) assert(false);
    if( !fs.load_source( src_frag ) ) assert(false);

    g_shader = new Shader_prog(vs, fs);
    g_shader->link();

    is_shaders_init = true;
}

}// END SHADER PICKING =========================================================

GlPick_FBO::GlPick_FBO(int width, int height) :
    _is_pick_init(false)
{
    _fbo = new GlFbo(false, GlFbo::COLOR|GlFbo::DEPTH);

    _color_tex = new GlTex2D(width, height, 0, GL_NEAREST, GL_CLAMP, GL_RG32UI );
    _color_tex->bind();
    _color_tex->allocate( GL_UNSIGNED_INT, GL_RG_INTEGER );

    _depth_tex = new GlTex2D(width, height, 0, GL_NEAREST, GL_CLAMP, GL_DEPTH_COMPONENT);
    _depth_tex->bind();
    _depth_tex->allocate( GL_FLOAT, GL_DEPTH_COMPONENT );
    GlTex2D::unbind();

    _fbo->bind();
    _fbo->set_render_size(width, height);
    _fbo->record_tex_attachment( GL_COLOR_ATTACHMENT0, _color_tex->id() );
    _fbo->record_tex_attachment( GL_DEPTH_ATTACHMENT , _depth_tex->id() );
    _fbo->update_attachments();
    assert( _fbo->check() );
    _fbo->unbind();

    if( !Shader_picking::is_shaders_init )
        Shader_picking::init_shaders();

    Shader_picking::nb_inst++;
}

// -----------------------------------------------------------------------------

GlPick_FBO::~GlPick_FBO()
{
    delete _fbo;
    delete _color_tex;
    delete _depth_tex;
    _fbo       = 0;
    _color_tex = 0;
    _depth_tex = 0;
    Shader_picking::nb_inst--;
    if( Shader_picking::nb_inst == 0)
        Shader_picking::delete_shaders();
}

// -----------------------------------------------------------------------------

void GlPick_FBO::begin(int x, int y, GLfloat* mvp_mat)
{
    _x = x;
    _y = _fbo->height() - y;

    glAssert( glClearColor( 0, 0, 0, 0 ) );
    _fbo->use_as_target();
    _fbo->clear_buffers(GlFbo::COLOR|GlFbo::DEPTH);

    Shader_picking::g_shader->use();
    Shader_picking::g_shader->set_mat4x4("MVP", mvp_mat);

    _is_pick_init = true;
}

// -----------------------------------------------------------------------------

GlPick_FBO::Ids GlPick_FBO::end()
{
    assert( _is_pick_init );
    _is_pick_init = false;
    Shader_picking::g_shader->unuse();

    unsigned tmp[20];

    _fbo->unbind();
    glAssert( glBindFramebufferEXT(GL_READ_FRAMEBUFFER, _fbo->id()) );
    glAssert( glReadBuffer(GL_COLOR_ATTACHMENT0) );
    glAssert( glReadPixels(_x, _y, 1, 1, GL_RG_INTEGER, GL_UNSIGNED_INT, tmp) );

    glAssert( glReadBuffer(GL_NONE) );
    glAssert( glBindFramebufferEXT(GL_READ_FRAMEBUFFER, 0) );

    GlPick_FBO::Ids id;
    // Compensate the minus 1 in fragment shader
    id._name         = (int)tmp[0] - 1;
    id._primitive_id = (int)tmp[1] - 1;
    return id;
}

// -----------------------------------------------------------------------------

void GlPick_FBO::set_name(unsigned id)
{
    assert( _is_pick_init );
    Shader_picking::g_shader->set_uniform("g_object_idx", (int)id);
}

// -----------------------------------------------------------------------------

void GlPick_FBO::resize(int w, int h)
{
    _color_tex->bind();
    _color_tex->set_size(w, h);
    _color_tex->allocate( GL_UNSIGNED_INT, GL_RED_INTEGER );

    _depth_tex->bind();
    _depth_tex->set_size(w, h);
    _depth_tex->allocate( GL_FLOAT, GL_DEPTH_COMPONENT );
    GlTex2D::unbind();

    _fbo->bind();
    _fbo->set_render_size(w, h);
    _fbo->update_attachments( true );
    _fbo->unbind();
}

// -----------------------------------------------------------------------------
