/*
 * This software is governed by the CeCILL-B license under French law and
 * abiding by the rules of distribution of free software.  You can  use, 
 * modify and/ or redistribute the software under the terms of the CeCILL-B
 * license as circulated by CEA, CNRS and INRIA at the following URL
 * "http://www.cecill.info" or the LICENCE.txt file present in this project.
*/

#ifndef GLPICK_FBO_HPP__
#define GLPICK_FBO_HPP__

#include "port_glew.h"

class GlFbo;
class GlTex2D;

/**
 * @name GlPick_FBO
 * @brief Handling opengl picking with FBO
 *
 * GlPick_FBO helps selecting objects into a 3d scene displayed with openGl.
 * To find which object is under the mouse GlPick_FBO will draw each object
 * with a specific color into an offline render buffer (the FBO).
 *
 * Use case:
 * @code
 * GlPick_FBO pick(width, height);
 * pick.begin(int x, int y, GLfloat* mvp_mat);
 * // The the associated index to the next drawned object
 * pick.set_name( obj_index );
 *
 * // Draw object with openGL
 *
 * int selected_obj = pick.end(); // retreive selected index if any *
 * @endcode
 *
 * @note don't forget to resize the FBO with resize() when the viewport
 * resolution changes
 *
 * @warning you must disable the shaders when drawing because GLPick_FBO use
 * its own shaders to draw the object
 *
*/
class GlPick_FBO {
public:

    struct Ids {
        int _name;         ///< name of the picked object (user defined)
        int _primitive_id; ///< primitive (triangles, points...) draw index
    };

    /// (width, height) = resolution of the current viewport
    GlPick_FBO(int width, int height);

    ~GlPick_FBO();

    /// Setup/enable FBO for drawing and enable dedicated shader
    void begin(int x, int y, GLfloat* mvp_mat);

    /// Disable FBO and shaders
    /// @return the picked primitive id and name or -1 if nothing is picked
    Ids end();

    /// Setup the primitive id. every drawing after this call will be associated
    /// with the integer 'id'
    /// @warning must be called between begin() end()
    void set_name(unsigned id);

    bool is_pick_init() const { return _is_pick_init; }

    /// Resize the FBO
    void resize(int w, int h);

private:
    GlFbo*   _fbo;       ///< Fbo used to render primitives to pick
    GlTex2D* _color_tex; ///< Color texture attached to the FBO
    GlTex2D* _depth_tex; ///< Depth texture attached to the FBO

    int _x, _y;          ///< pick position set with begin().
    bool _is_pick_init;  ///< Wether we're inside a begin() end()
};

#endif // GLPICK_FBO_HPP__
