/*
 * This software is governed by the CeCILL-B license under French law and
 * abiding by the rules of distribution of free software.  You can  use, 
 * modify and/ or redistribute the software under the terms of the CeCILL-B
 * license as circulated by CEA, CNRS and INRIA at the following URL
 * "http://www.cecill.info" or the LICENCE.txt file present in this project.
*/

#ifndef INTERSECTION_H_
#define INTERSECTION_H_

/**
  @namespace Inter
  @brief Various intersection calculus compatible CUDA nvcc and GCC.

*/

#include "vec3_cu.hpp"
#include "point_cu.hpp"

// =============================================================================
namespace Inter {
// =============================================================================

typedef Vec3_cu Vec;
typedef Vec3_cu Point;

// -----------------------------------------------------------------------------

struct Line {

    IF_CUDA_DEVICE_HOST
    Line(Vec dir, const Point& org) {
        this->dir = dir.normalized();
        this->org = org;
    }

    Vec   dir;  ///< Line direction
    Point org;  ///< Line origine
};

// -----------------------------------------------------------------------------

struct Plane {

    IF_CUDA_DEVICE_HOST
    Plane(Vec normal, const Point& org){
        this->normal = normal.normalized();
        this->org    = org;
    }

    Vec   normal;  ///< Normale to plane
    Point org;     ///< Plane origine
};

// -----------------------------------------------------------------------------

struct Sphere {

    IF_CUDA_DEVICE_HOST
    Sphere(float radius, const Point& org){
        this->radius = radius;
        this->org    = org;
    }

    float radius; ///< sphere radius
    Point org;    ///< sphere origine
};

// -----------------------------------------------------------------------------

/// A cylinder start from org and stop at org+dir*length
struct Cylinder {

    IF_CUDA_DEVICE_HOST
    Cylinder(float radius, float length, const Vec& dir, const Point& org){
        this->radius = radius;
        this->org    = org;
        this->dir    = dir.normalized();
        this->length = length;
    }

    float length; ///< cylinder length
    float radius; ///< cylinder radius
    Point org;    ///< cylinder origine
    Vec   dir;    ///< cylinder direction
};

// -----------------------------------------------------------------------------

struct Triangle {

    IF_CUDA_DEVICE_HOST
    Triangle(const Point& p0, const Point& p1, const Point& p2){
        p[0] = p0; p[1] = p1; p[2] = p2;
    }

    Point p[3];
};

// -----------------------------------------------------------------------------

/// @param res  intersection location
/// @return true if intersect false otherwise
IF_CUDA_DEVICE_HOST static inline
bool plane_line(const Plane& p, const Line& l, Point& res);

/// @param res  nearest intersection location
IF_CUDA_DEVICE_HOST static inline
bool sphere_line(const Sphere& s, const Line& l, Point& res, float& t);

/// @param res  nearest intersection location
IF_CUDA_DEVICE_HOST static inline
bool cylinder_line(const Cylinder& c, const Line& l, Point& res, float& t);

/// self intersection of a line with a line
/// @param res    the intersection position if there is any
/// @return true  if intersect
IF_CUDA_DEVICE_HOST static inline
bool line_line( const Line& line1, const Line& line2, Point& res);

/// self intersection of a triangle with a triangle
/// @param res1, res2 : the intersection segment if there is any
/// @param coplanar : if triangles are coplanar
/// @param eps : the chosen epsilon threshold to detect intersection.
/// set to zero to disable this test (but less robust).
/// @return true if intersects
/// @note If you wish to find intersection between two mesh or self-intersection
/// you might want to avoid degenerate cases. To do so add some random noise
/// to your vertex positions. The noise norm must be greater than 'eps' and it
/// will statistically guarantee to find a closed curve of intersection (if the
/// mesh are closed and manifold). Otherwise you'll have to do more testing,
/// knowing that it can take a while to produce a robust algorithm ...
IF_CUDA_DEVICE_HOST static inline
bool tri_tri(const Triangle& tri1, const Triangle& tri2,
             Point& res1, Point& res2,
             bool& coplanar,
             float eps = 0.000001f);

/// intersection of the sphere s and the segment at p in direction d
/// returns the number of intersections: 0, 1, or 2.
/// also returns in it1 and it2 the parameter value, along the segment, at which
/// the intersections occur.
/// For convolution surfaces
IF_CUDA_DEVICE_HOST
static inline int sphere_segment(const Sphere& s,
                                 const Point_cu& p,
                                 const Vec3_cu& d,
                                 float& it1,
                                 float& it2);

/// intersection of the sphere s and the segment at p in direction d
/// returns the number of intersections: 0, 1, or 2.
/// also returns in it1 and it2 the parameter value, along the segment, at which
/// the intersections occur.
/// For homothetic convolution surfaces
IF_CUDA_DEVICE_HOST
static inline bool sphere_segment_clamped(const Sphere& s,
                                          const Point_cu& ori,
                                          const Vec3_cu& d,
                                          float len,
                                          float w1,
                                          float w2,
                                          float& it1,
                                          float& it2);

}// Inter ======================================================================

#include "intersection.inl"

#endif // INTERSECTION_H_



