/*
 * This software is governed by the CeCILL-B license under French law and
 * abiding by the rules of distribution of free software.  You can  use, 
 * modify and/ or redistribute the software under the terms of the CeCILL-B
 * license as circulated by CEA, CNRS and INRIA at the following URL
 * "http://www.cecill.info" or the LICENCE.txt file present in this project.
*/

#include "intersection.hpp"
#include "inter_tri_tri.inl"

// =============================================================================
namespace Inter {
// =============================================================================

IF_CUDA_DEVICE_HOST static inline
bool plane_line(const Plane& p, const Line& l, Point& res)
{

    /*
        Equation du plan :
            a*x + b*y + c*z + d = 0

        normale du plan : (a b c)
        parametre d = ( -a xa -b ya -c za) avec (xa ya za) un point du plan

        Equation parametrique de la droite de parametre t
            x = x1 + i*t
            y = y1 + j*t
            z = z1 + k*t

        Vecteur directeur (i j k)
        Point de la droite (x1 y1 z1)

        On peut calculer t :

        t = - (a*x1 + b*y1 + c*z1 + d)/(a*i + b*j + c*k)

        denominateur = 0 -> pas de solution.
        Cas particulier :
        denominateur = 0 et numerateur = 0 et x1 y1 z1 appartient
        au plan (a*x + b*y + c*z + d = 0)
        Alors la droite est contenue dans ce plan
    */
    const float eps    = 0.00001f;
    const Vec p_normal = p.normal;
    const Vec p_org    = p.org;
    const Vec l_dir    = l.dir;
    const Point l_org  = l.org;

    const float denominator = p_normal.dot( l_dir );

    const float a = fabs(denominator);
    if(a < eps) return false;

    float d = -(p_normal.dot(p_org));
    float t = -(p_normal.dot(l_org) + d);
    t /= denominator;

    res = l_org + l_dir * t;

    return true;
}

// -----------------------------------------------------------------------------

IF_CUDA_DEVICE_HOST static inline
bool sphere_line(const Sphere& s, const Line& l, Point& res, float& parameter){

    Point center = s.org;
    Point orig   = l.org;
    Vec   dir    = l.dir;

    float a = dir.norm_squared();
    float b = 2.0f * (dir.dot(orig - center));
    float c = (orig - center).norm_squared() - s.radius*s.radius;
    float d = b*b - 4.0f*a*c;

    if (d >= 0){
        d = sqrt(d);

        float t1 = (-b-d) / (2.0f*a);
        float t2 = (-b+d) / (2.0f*a);

        float t  = 1.00001f;
        if (t1 > t2) t = t2;
        else         t = t1;

        res       = orig + dir*t;
        parameter = t;
        return true;
    }
    return false;
}

// -----------------------------------------------------------------------------

IF_CUDA_DEVICE_HOST static inline
bool cylinder_line(const Cylinder& cy, const Line& l, Point& res, float& t){
    // Calculer l'intersection du segment p0 p1 le cylindre est une
    // equation du second degres
    // t^2 (v x u)^2 + t * 2 * (v x u) * ((o-p) x u) + ((o-p) x u)^2 - R^2 = 0
    // Avec :
    // v : vecteur directeur du segment
    // u : vecteur directeur du cylindre
    // o : origine du segment
    // p : origine du cylindre
    // t : parametre de l'equation parametrique du segment
    const float eps = 0.0000001f;

    Vec POxU = (l.org - cy.org).cross(cy.dir);
    Vec VxU  = l.dir.cross(cy.dir);

    float a = VxU. dot( VxU  );
    float b = VxU. dot( POxU ) * 2.0f;
    float c = POxU.dot( POxU ) - cy.radius * cy.radius;

    // Determinant de l'equation :
    float d    = b*b - 4.0f*a*c;

    if (d >= 0)
    {
        d = sqrt(d);

        if(a < eps)
            return false;

        float t1 = (-b-d) / (2.0f*a);
        float t2 = (-b+d) / (2.0f*a);
        // On cherche la distance la plus courte :
        t  = 1.00001f;
        if (t1 > t2) t = t2;
        else         t = t1;

        res = l.org + l.dir*t;

        float proj_length = cy.dir.dot( res - cy.org );

        if(proj_length < 0.f || proj_length > cy.length)
            return false;

        return true;
    }
    return false;
}

// -----------------------------------------------------------------------------

IF_CUDA_DEVICE_HOST static inline
bool line_line( const Line& line1, const Line& line2, Point& res)
{
    // doctor math :
    // Let's try this with vector algebra. First write the two equations like
    // this.

    // L1 = P1 + a V1
    // L2 = P2 + b V2

    // P1 and P2 are points on each line. V1 and V2 are the direction vectors
    // for each line.

    // If we assume that the lines intersect, we can look for the point on L1
    // that satisfies the equation for L2. This gives us this equation to
    // solve.

    // P1 + a V1 = P2 + b V2

    // Now rewrite it like this.

    // a V1 = (P2 - P1) + b V2

    // Now take the cross product of each side with V2. This will make the
    // term with 'b' drop out.

    // a (V1 X V2) = (P2 - P1) X V2

    // If the lines intersect at a single point, then the resultant vectors
    // on each side of this equation must be parallel, and the left side must
    // not be the zero vector. We should check to make sure that this is
    // true. Once we have checked this, we can solve for 'a' by taking the
    // magnitude of each side and dividing. If the resultant vectors are
    // parallel, but in opposite directions, then 'a' is the negative of the
    // ratio of magnitudes. Once we have 'a' we can go back to the equation
    // for L1 to find the intersection point.

    const float eps = 0.00001f;

    Vec V1xV2   = line1.dir.cross(line2.dir);

    // Denominateur nulle alors pas de solution (droites colineaires)
    float V1xV2norm = V1xV2.norm();
    if(V1xV2norm < eps)
        return false;

    Vec P1P2xV2 = (line2.org - line1.org).cross( line2.dir );

    // Verifie si elle sont contenues dans le meme plan :
    if( (P1P2xV2.cross(V1xV2)).norm() > eps )
        return false;

    float a = P1P2xV2.norm() / V1xV2norm;
    // On determine le signe de 'a' en verifiant que les vecteurs sont ou non
    // de sens opposes :
    if( P1P2xV2.dot(V1xV2) > 0)
        res = line1.org + line1.dir *  a;
    else
        res = line1.org + line1.dir * -a;

    return true;
}

// -----------------------------------------------------------------------------

IF_CUDA_DEVICE_HOST static inline
bool tri_tri( const Triangle& tri1, const Triangle& tri2,
              Point& res1, Point& res2,
              bool& cop, float eps)
{
    cop = false;
    bool s = Inter_tri::tri_tri_intersect_with_isectline(
                (float*)&(tri1.p[0]), (float*)&(tri1.p[1]), (float*)&(tri1.p[2]),
                (float*)&(tri2.p[0]), (float*)&(tri2.p[1]), (float*)&(tri2.p[2]),
                cop,
                (float*)&res1, (float*)&res2,
                eps
                );
    return s;

}

// -----------------------------------------------------------------------------

IF_CUDA_DEVICE_HOST
static inline int sphere_segment(const Sphere& s,
                                 const Point_cu& p,
                                 const Vec3_cu& d,
                                 float& it1,
                                 float& it2)
{
    float a = d.norm_squared();
    float b = 0.0;
    float c = -s.radius*s.radius;
    for(unsigned int i = 0; i<3; ++i){
        float aux = p[i]-s.org[i];
        b += d[i]*aux;
        c += aux*aux;
    }
    b *= 2.0;

    float delta = b*b - 4.0*a*c;

    if(delta<0.0) {
        return 0;
    } else if (delta==0.0) {
        it1 = -b/(2.0*a);
        return 1;
    } else {
        float sqrt_delta = sqrt(delta);
        it1 = (-b-sqrt_delta)/(2.0*a);
        it2 = (-b+sqrt_delta)/(2.0*a);
        return 2;
    }
}

// -----------------------------------------------------------------------------

IF_CUDA_DEVICE_HOST
static inline bool sphere_segment_clamped(const Sphere& s,
                                          const Point_cu& ori,
                                          const Vec3_cu& dir,
                                          float len,
                                          float w1,
                                          float w2,
                                          float& it1,
                                          float& it2)
{
    const Vec3_cu p1_to_center = s.org - ori;

    const float warped_radius_2 = s.radius * s.radius;

    // On cherche t /in [0,1] tq at^2-2bt+c <= 0
    float       a = len*len - warped_radius_2 * w2*w2;
    const float b = dir.dot(p1_to_center) + warped_radius_2 * w2*w1;
    const float c = p1_to_center.norm_squared()  - warped_radius_2 * w1*w1;

    // On étudie l'inequation a*t^2 - 2*b*t + c <= 0

    // WARNING : instable lorsque a est au voisinage de 0 (division par a....)
    if(fabs(a) < 0.0000001){
        a = 0.0;
    }
    if((b*b - a*c) == 0.0 && a==0.0 && b==0.0)
        return false;

    float delta = b*b - a*c;
    if(delta>0.0) {
        //la droite prolongeant le segment intersecte la "sphère" en 2 points => faux : il reste le cas dégénéré (affine !)

        //La signification de ces points depend du signe de a !!!!!
        if(a>0.0) {// sans doute a remplacer par > 0.00000001
            // Dans ce cas on clippe une unique portion de segment !!!!
            // celle-ci est situé entre les deux racines de l'équation

            // on doit donc avoir l1 < l2 !!!!
            delta = sqrt(delta);
            float l_pos1 = (b - delta)/a;
            float l_pos2 = (b + delta)/a;

            if(l_pos2 <= 0.0 || l_pos1 >= 1.0) {
                // pas vraiment utile si l'on utilise un gestionnaire de segment !!!
                it1 = 1.0;
                it2 = 0.0;
                return false; // pas de partie clippé
            }

            l_pos1 = (l_pos1<0.0) ? 0.0 : l_pos1 ;
            l_pos2 = (l_pos2<1.0) ? l_pos2 : 1.0 ;

            it1 = l_pos1;
            it2 = l_pos2;
            return true;
        } else if(a<0.0) { // sans doute a remplacer par < -0.00000001
            // Dans ce cas on clippe potentiellement deux portions de segment !!!!
            // celles-ci sont situés de part et d'autre des racines de l'équation

            // l1 > l2
            delta = sqrt(delta);
            float l_pos1 = (b - delta)/a;
            float l_pos2 = (b + delta)/a;

            if(l_pos2 <= 0.0 && l_pos1 >= 1.0) {
                // pas vraiment utile si l'on utilise un gestionnaire de segment !!!
                it1 = 1.0;
                it2 = 0.0;
                return false; // pas de partie clippé
            }
            if(l_pos1 < 0.0 || l_pos2 > 1.0) {
                // On clippe tout le segment
                it1 = 0.0;
                it2 = 1.0;
                return true;
            }
            if(l_pos2 <= 0.0) {
                // on a alors necessairement 0.0 < l_pos1 < 1.0
                l_pos2 = 1.0;
                it1 = l_pos1;
                it2 = l_pos2;
                return true;
            }
            if(l_pos1 >= 1.0) {
                // on a alors necessairement 0.0 < l_pos2 < 1.0
                l_pos1 = 0.0;
                it1 = l_pos1;
                it2 = l_pos2;
                return true;
            }
            // c'est le seul cas ou l'on divise le segment en 2 portions
            // et donc ou l_pos1 et l_pos2 sont inversé
            it1 = l_pos1;
            it2 = l_pos2;
            return true;    //TODO : @todo : Normalement cas impossible ... ???
        } else {
            // a = 0 => on est dans le cas affine !!!
            // => comportement dépend du signe de b

            // la racine est unique
            if(-2.0*b>0.0) {
                // On clippe ce qui est a gauche
                float l_pos = c / (2.0*b);

                if(l_pos>=1.0) {
                    // On clippe tout le segment
                    it1 = 0.0;
                    it2 = 1.0;
                    return true;
                } else if(l_pos<=0.0) {
                    // pas vraiment utile si l'on utilise un gestionnaire de segment !!!
                    it1 = 1.0;
                    it2 = 0.0;
                    // on ne clippe rien
                    return false;
                } else {
                    it1 = 0.0;
                    it2 = l_pos;
                    return true;
                }
            } else if(-2.0*b<0.0) {
                // On clippe ce qui est a droite
                float l_pos = c / (2.0*b);

                if(l_pos<=0.0) {
                    // On clippe tout le segment
                    it1 = 0.0;
                    it2 = 1.0;
////                    return this->seg_.length();
                    return true;
                } else if(l_pos>=1.0) {
                    // pas vraiment utile si l'on utilise un gestionnaire de segment !!!
                    it1 = 1.0;
                    it2 = 0.0;
                    // on ne clippe rien
                    return false;
                } else {
                    it1 = l_pos;
                    it2 = 1.0;
////                    return this->seg_.length() * (1.0-l_pos);
                    return true;
                }
            } else {
                //NB non atteignable à cause du test sur delta !!!!
                // b = 0.0
                if(c > 0.0) {
                    // pas vraiment utile si l'on utilise un gestionnaire de segment !!!
                    it1 = 1.0;
                    it2 = 0.0;
                    //on ne clippe rien
                    return false;
                } else {
                    // on clippe tout
                    it1 = 0.0;
                    it2 = 1.0;
//                    return this->seg_.length();
                    return true;
                }
            }
        }
    }

    //TODO : @todo
    // Has bee added to prevent a warning : should be check if this is the value to be returned in this case

    // Il y a un cas ou cela est faux : le cas ou delta = 0 et a < 0
    it1 = 1.0;
    it2 = 0.0;
    return false;
    //TODO : @todo
    // pas vraiment utile si l'on utilise un gestionnaire de segment !!!
}

}// END INTER NAMESPACE ========================================================
