/*
 * This software is governed by the CeCILL-B license under French law and
 * abiding by the rules of distribution of free software.  You can  use, 
 * modify and/ or redistribute the software under the terms of the CeCILL-B
 * license as circulated by CEA, CNRS and INRIA at the following URL
 * "http://www.cecill.info" or the LICENCE.txt file present in this project.
*/

#ifndef TRS_HPP__
#define TRS_HPP__

#include "vec3_cu.hpp"
#include "transfo.hpp"

/**
    @class TRS
    @brief Compact representation of (Translation * Rotation * Scale) matrix

*/
struct TRS {

    /// Default constructor no transformation
    TRS() :
        _axis( Vec3_cu::unit_x() ),
        _angle( 0.f ),
        _scale( Vec3_cu::unit_scale() ),
        _translation( Vec3_cu::zero() )
    {  }

    TRS(const Vec3_cu& trans, const Vec3_cu& axis, float angle, const Vec3_cu& scale) :
        _axis( axis ),
        _angle( angle ),
        _scale( scale ),
        _translation( trans )
    {  }

    /// Build pure rotation
    static TRS rotation(const Vec3_cu& axis, float angle) {
        return TRS(Vec3_cu::zero(), axis, angle, Vec3_cu::unit_scale() );
    }

    /// Build pure translation
    static TRS translation(const Vec3_cu& trans) {
        return TRS(trans, Vec3_cu::unit_x(), 0.f, Vec3_cu::unit_scale() );
    }

    /// Build pure scale
    static TRS scale(const Vec3_cu& scale) {
        return TRS(Vec3_cu::zero(), Vec3_cu::unit_x(), 0.f, scale );
    }

    Transfo to_transfo() {
        return Transfo::translate(_translation) *
               Transfo::rotate(_axis, _angle) *
               Transfo::scale(_scale);
    }

    // -------------------------------------------------------------------------
    /// @name Attributes
    // -------------------------------------------------------------------------

    Vec3_cu _axis;         ///< Axis of rotation
    float   _angle;        ///< angle of rotation
    Vec3_cu _scale;        ///< Scale in (x,y,z) directions
    Vec3_cu _translation;  ///< Vector of translation
};

#endif // TRS_HPP__
