/*
 * This software is governed by the CeCILL-B license under French law and
 * abiding by the rules of distribution of free software.  You can  use, 
 * modify and/ or redistribute the software under the terms of the CeCILL-B
 * license as circulated by CEA, CNRS and INRIA at the following URL
 * "http://www.cecill.info" or the LICENCE.txt file present in this project.
*/

#ifndef STD_UTILS_HPP__
#define STD_UTILS_HPP__

#include <cassert>
#include <vector>
#include <list>
#include <map>
#include <sstream>
#include <algorithm>

/**
    @namespace Std_utils
    @brief Utilities for the stl std::vector std::map etc.

    usage:
    @code
        // Signature nomenclature :
        bool exists(std::container c, ELT) // return true if 'ELT' is in 'c'
        bool find(std::container c, ELT, RES&) // return true if 'ELT' is in 'c'
        RES& find(std::container, ELT)
    @endcode

*/

// TODO: explode std utils in several headers (std_utils_vector/list/map etc.)
// =============================================================================
namespace Std_utils {
// =============================================================================

/// Convert a scalar (int float double long unsigned etc.) to a string
/// @warning no type checking is done
// Possibly later we could specialized the template
template<typename T>
static std::string to_string(T number)
{
   std::stringstream ss;
   ss << number;
   return ss.str();
}

// -----------------------------------------------------------------------------

/// Converting a string to real value.
/// the template parameter will not be infered call with this syntax:
/// @code
///     float v = Std_utils::to_real<float>( str_val );
///     int   v = Std_utils::to_real<int>  ( str_val );
/// @endcode
template<typename T>
static T to_real(const std::string& number)
{
    std::stringstream ss( number );
    T real;
    ss >> real;
    return real;
}

// -----------------------------------------------------------------------------

/// @return lower case version of the string 'str'
inline static std::string to_lower(const std::string& str){
    std::string s = str;
    std::transform(s.begin(), s.end(), s.begin(), ::tolower);
    return s;
}

// -----------------------------------------------------------------------------

/// @return upper case version of the string 'str'
inline static std::string to_upper(const std::string& str){
    std::string s = str;
    std::transform(s.begin(), s.end(), s.begin(), ::toupper);
    return s;
}

// -----------------------------------------------------------------------------

/// @return the file extenssion or the empty string if not found
inline static std::string file_ext(const std::string& str){
    std::string res;
    unsigned pos = str.find_last_of('.');
    res = pos < str.size() ? str.substr(pos) : "";
    return res;
}

// -----------------------------------------------------------------------------

/// Pops the ith element by swapping the last element of the vector with it
/// and decrementing the size of the vector
template <class T>
static void pop(std::vector<T>& vec, int i)
{
    assert(vec.size() > 0);
    vec[i] = vec[vec.size() - 1];
    vec.pop_back();
}

// -----------------------------------------------------------------------------

/// Concatenate v0 and v1 into v0. their types can be different as long as
/// T0 and T1 are equal in terms of byte size.
template<class T0, class T1>
static void concat(std::vector<T0>& v0, std::vector<T1>& v1)
{
    assert(sizeof(T0) == sizeof(T1));
    v0.resize( v0.size() + v1.size());
    const int off = v0.size();
    for(unsigned i = 0; i < v1.size(); i++)
        v0[off + i] = *( reinterpret_cast<T0*>( &(v1[i]) ) );
}

// -----------------------------------------------------------------------------

/// @return true if v0 and v1 are equal in size and their elements match
template<class T0, class T1>
static bool equal(std::vector<T0>& v0, std::vector<T1>& v1)
{
    if(v0.size() != v1.size())
        return false;

    for(unsigned i = 0; i < v1.size(); ++i)
        if(v0[i] != v1[i]) return false;

    return true;
}

// -----------------------------------------------------------------------------

/// Copy src into dst (dst is resized ). their types can be different as long as
/// T0 and T1 are equal in terms of byte size.
template<class T0, class T1>
static void copy(std::vector<T0>& dst, const std::vector<T1>& src)
{
    assert(sizeof(T0) == sizeof(T1));
    dst.resize( src.size() );
    for(unsigned i = 0; i < src.size(); i++)
        dst[i] = *( reinterpret_cast<const T0*>( &(src[i]) ) );
}

// -----------------------------------------------------------------------------

/// Find 'elt' in 'vec'. Search is O(n).
template<class T0, class T1>
static bool exists(const std::vector<T0>& vec, const T1& elt)
{
    for(unsigned i = 0; i < vec.size(); i++)
        if(vec[i] == elt) return true;

    return false;
}

// -----------------------------------------------------------------------------

/// Find 'elt' in 'vec'. Search is O(n).
/// @return the index of the first occurence of elt
template<class T0, class T1>
static int find(const std::vector<T0>& vec, const T1& elt)
{
    for(unsigned i = 0; i < vec.size(); i++)
        if(vec[i] == elt) return i;

    return -1;
}

// -----------------------------------------------------------------------------

/// Erase an element at the 'ith' position.
template<class T0>
static void erase(std::vector<T0>& vec, int ith)
{
    vec.erase( vec.begin() + ith);
}

// -----------------------------------------------------------------------------

/// Find 'elt' in 'list'. Search is O(n).
template<class T0, class T1>
static bool exists(const std::list<T0>& list, const T1& elt)
{
    typename std::list<T0>::const_iterator it = list.begin();
    for(; it != list.end(); ++it)
        if( (*it) == elt) return true;

    return false;
}

// -----------------------------------------------------------------------------

/// Find and retreive an element from the map. If not found an assertion is
/// triggered
/// @param elt : the element to be found
/// @return what is associated with 'k' in 'map'.
// third template parameter is here to avoid ambiguities
template<class Key, class Elt, class PKey>
static const Elt& find(const std::map<Key, Elt>& map, const PKey& k)
{
    typename std::map<Key, Elt>::const_iterator it = map.find( k );
    if( it != map.end() )
        return it->second;
    else
    {
        assert( false );
        return map.begin()->second;
    }
}

// -----------------------------------------------------------------------------

/// Find and retreive an element from the map. If not found an assertion is
/// triggered
/// @param elt : the element to be found
/// @return what is associated with 'k' in 'map'.
// third template parameter is here to avoid ambiguities
template<class Key, class Elt, class PKey>
static Elt& find(std::map<Key, Elt>& map, const PKey& k)
{
    typename std::map<Key, Elt>::iterator it = map.find( k );
    if( it != map.end() )
        return it->second;
    else
    {
        assert( false );
        return map.begin()->second;
    }
}

// -----------------------------------------------------------------------------

/// Find and retreive an element from the map.
/// @param elt : the key element to found
/// @param res : what is associated with 'elt' in 'map'.
/// @return if we found the element
// third/fourth templates parameters are there to avoid ambiguities
template<class Key, class Elt, class PKey, class PElt>
static bool find(const std::map<Key, Elt>& map, const PKey& elt, PElt const * & res)
{
    typename std::map<Key, Elt>::const_iterator it = map.find( elt );
    if( it != map.end() )
    {
        res = &(it->second);
        return true;
    }
    else
    {
        res = 0;
        return false;
    }
}

// -----------------------------------------------------------------------------

/// Find and retreive an element from the map.
/// @param elt : the key element to found
/// @param res : what is associated with 'elt' in 'map'.
/// @return if we found the element
// third/fourth templates parameters are there to avoid ambiguities
template<class Key, class Elt, class PKey, class PElt>
static bool find(std::map<Key, Elt>& map, const PKey& elt, PElt*& res)
{
    typename std::map<Key, Elt>::iterator it = map.find( elt );
    if( it != map.end() )
    {
        res = &(it->second);
        return true;
    }
    else
    {
        res = 0;
        return false;
    }
}

// -----------------------------------------------------------------------------

/// Find and retreive an element from the map.
/// @param elt : the key element to found
/// @param res : what is associated with 'elt' in 'map'.
/// @return if we found the element
// third template parameter is here to avoid ambiguities
template<class Key, class Elt, class PKey>
static bool exists(const std::map<Key, Elt>& map, const PKey& elt)
{
    return map.find( elt ) != map.end();
}

}// End Namespace Std_utils ====================================================

#include "cuda_compiler_interop.hpp"

/**
    @namespace Utils
    @brief Utilities for the c++ programmer everyday life (libs independant)
*/
// =============================================================================
namespace Utils {
// =============================================================================

/// Function macro to generically swap datas or pointers in a single line
template<class T>
IF_CUDA_DEVICE_HOST static inline
void swap(T& src, T& dst){
    T tmp = src;
    src = dst;
    dst = tmp;
}

// -----------------------------------------------------------------------------

/// Simple recopy of two arrays of arbitrary type.
template<class T>
void copy(T* dst, const T* src, int nb_elt)
{
    for (int i = 0; i < nb_elt; ++i)
        dst[i] = src[i];
}

// -----------------------------------------------------------------------------

/// A simple integer power function
/// @return x^p
/// @note p must be positive
template<class T>
static inline
T ipow(T x, int p)
{
    assert(p >= 0);
    if (p == 0) return 1;
    if (p == 1) return x;
    return x * ipow(x, p-1);
}

// -----------------------------------------------------------------------------

/// A static version of the integral power
/// @return a^n
template <int n> inline
float ipow(float a) {
    const float b = ipow<n/2>(a);
    return (n & 1) ? (a * b * b) : (b * b);
}

template <> inline float ipow<1>(float a){ return a;   }
template <> inline float ipow<0>(float  ){ return 1.f; }

// -----------------------------------------------------------------------------

/// A static version of the integral power
/// @return a^n
template <int n> inline
int ipow(int a) {
    const int b = ipow<n/2>(a);
    return (n & 1) ? (a * b * b) : (b * b);
}

template <> inline int ipow<1>(int a){ return a; }
template <> inline int ipow<0>(int  ){ return 1; }

// -----------------------------------------------------------------------------


}// End Namespace Utils ========================================================

#endif // STD_UTILS_HPP__
