#include <stdio.h>
#include <stdlib.h>
#include <GL/glew.h>
#include <GL/glut.h>
#include <stdbool.h>

#include "vector4.h"
#include "material.h"
#include "camera.h"
#include "mesh.h"

static int _windowNumber;
int _angle  = 0;

bool _anim   = true;
bool _ortho  = false;
bool _normal = false;

float _width  = 800;
float _height = 600;

Camera _cam;
Mesh*  _mesh;

Materials plaster = {
    {0.2f, 0.f, 0.f, 0.f},
    {1.f , 0.f, 0.f, 0.f},
    {0.f , 0.f, 0.f, 0.f},
    0.f
};

Materials mirror = {
    {0.2f, 0.f, 0.f, 0.f},
    {0.f , 0.f, 0.f, 0.f},
    {1.f , 0.f, 0.f, 0.f},
    40.f
};

Materials pvc = {
    {0.2f, 0.f, 0.f, 0.f},
    {1.f , 0.f, 0.f, 0.f},
    {1.f , 1.f, 1.f, 0.f},
    25.f
};

Materials metal = {
    {0.2f, 0.f, 0.f, 0.f},
    {0.4f, 0.f, 0.f, 0.f},
    {1.0f, 0.f, 0.f, 0.f},
    25.f
};

void special_keys(int k, int mx, int my)
{
    // Avoid annoying warnings
    mx = my;
    my = mx;

    switch(k)
    {
    case GLUT_KEY_UP :
        _cam = camera_walk(_cam,  0.1f);
        break;
    case GLUT_KEY_DOWN :
        _cam = camera_walk(_cam, -0.1f);
        break;
    case GLUT_KEY_RIGHT :
        _cam = camera_walksideway(_cam, 0.1f);
        break;
    case GLUT_KEY_LEFT :
        _cam = camera_walksideway(_cam, -0.1f);
        break;
    }
    glutPostRedisplay();
}

/* Key events */
void key(unsigned char c, int mx, int my)
{
    // Avoid annoying warnings
    mx = my;
    my = mx;

    static bool depth      = true;
    static bool wire       = false;
    static bool light      = true;
    static bool light0     = true;
    static bool light1     = true;
    static bool light2     = true;
    static bool flat_shade = false;

    switch (c)
    {
    case'a':
        _anim = !_anim;
        break;
    case'o':
        _ortho = !_ortho;
    break;
    case'n':
        _normal = !_normal;
    break;
    case'z':
        if(depth) glDisable(GL_DEPTH_TEST);
        else      glEnable (GL_DEPTH_TEST);
        depth = !depth;
        break;
    case'w':
        if(wire) glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
        else     glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);
        wire = !wire;
        break;
    case'e':
        if(light) glDisable(GL_LIGHTING);
        else      glEnable(GL_LIGHTING);
        light = !light;
        break;
    case 'g':
        if(flat_shade) glShadeModel(GL_SMOOTH);
        else           glShadeModel(GL_FLAT);
        flat_shade = !flat_shade;
        break;
    case'0':
        if(light0) glDisable(GL_LIGHT0);
        else      glEnable(GL_LIGHT0);
        light0 = !light0;
        break;
    case'1':
        if(light1) glDisable(GL_LIGHT1);
        else      glEnable(GL_LIGHT1);
        light1 = !light1;
        break;
    case'2':
        if(light2) glDisable(GL_LIGHT2);
        else       glEnable(GL_LIGHT2);
        light2 = !light2;
        break;

    case 'i': _cam = camera_pitch(_cam,  0.1f); break;
    case 'k': _cam = camera_pitch(_cam, -0.1f); break;
    case 'j': _cam = camera_yaw  (_cam, -0.1f); break;
    case 'l': _cam = camera_yaw  (_cam,  0.1f); break;
        /* ESCAPE: leave application */
    case 27 :
        mesh_delete(_mesh);
        glFinish();
        glutDestroyWindow(_windowNumber);
        exit (0);
        break;
    }
    glutPostRedisplay();
}

void no_events()
{
    _angle = _angle+1 % 360;
    glutPostRedisplay();
}

void reshape(int x, int y)
{
    glViewport(0, 0, x, y);
    _width  = x;
    _height = y;
}

void init_light()
{
    glEnable(GL_NORMALIZE);
    glEnable(GL_LIGHTING);
    glEnable(GL_LIGHT0);
    glEnable(GL_LIGHT1);
    glEnable(GL_LIGHT2);

    int max_light;
    glGetIntegerv(GL_MAX_LIGHTS, &max_light);
    printf("Maximum number of lights %d\n", max_light);fflush(stdout);

    Vec4 light0Ambient, light0Diffuse, light0Specular;
    vec4_set(light0Ambient , 0.2f, 0.2f, 0.2f, 0.f);
    vec4_set(light0Diffuse , 1.0f, 1.0f, 1.0f, 0.f);
    vec4_set(light0Specular, 1.0f, 1.0f, 1.0f, 0.f);
    int i = 0;
    for(i = 0; i < 3; i++)
    {
        glLightfv(GL_LIGHT0+i, GL_AMBIENT , light0Ambient);
        glLightfv(GL_LIGHT0+i, GL_DIFFUSE , light0Diffuse);
        glLightfv(GL_LIGHT0+i, GL_SPECULAR, light0Specular);
    }

    glLightf(GL_LIGHT2, GL_SPOT_CUTOFF, 15);
    glLightf(GL_LIGHT2, GL_SPOT_EXPONENT, 100);
}

void set_torch()
{
    Vec4 light2_position;
    vec4_set(light2_position, 0.f, 0.f, 0.f, 1.f);
    glLightfv(GL_LIGHT2, GL_POSITION, light2_position);
    vec4_set(light2_position, 0.0f, 0.0f, -1.0f, 0.f);
    glLightfv(GL_LIGHT2, GL_SPOT_DIRECTION, light2_position);
}

void set_light_positions()
{
    Vec4 light0_position;
    vec4_set(light0_position, -3.f, 1.f, 0.f, 1.f);
    glLightfv(GL_LIGHT0, GL_POSITION, light0_position);
    Vec4 light1_position;
    vec4_set(light1_position, 0.f, 2.f, 4.f, 1.f);
    glLightfv(GL_LIGHT1, GL_POSITION, light1_position);
}

/* Rendering function */
void display(void)
{
    glClearColor(0.f, 0.f, 0.f, 0.f);
    glClear(GL_COLOR_BUFFER_BIT|GL_DEPTH_BUFFER_BIT);

    glMatrixMode(GL_PROJECTION);
    glLoadIdentity();
    if(_ortho) glOrtho(-5.f, 5.f, -5.f, 5.f, -5.f, 5.f );
    else       gluPerspective(60.f, _width/_height, 0.1f, 100.f);

    glMatrixMode(GL_MODELVIEW);
    glLoadIdentity();

    set_torch();
    camera_place(_cam);

    set_light_positions();

    glColor3f(1.f, 0.f, 0.f);
    materials_set_opengl_state(metal);
    glutSolidCube(1.f);

    glPushMatrix();
    if(_anim){
        glRotatef(_angle, 0.f, 1.f, 0.f);
        glTranslatef(3.f, 0.f, 0.f );
    }else
        glTranslatef(0.f, 1.23f, 0.f );

    glColor3f(1.f, 1.f, 0.f);
    glutSolidTeapot(1.f);
    glPopMatrix();

    glPushMatrix();
    glScalef(3.f, 3.f, 3.f);
    glTranslatef(0.f, 0.5f, 0.f);
    mesh_draw(_mesh);
    if(_normal) mesh_draw_normals(_mesh, 0.01f);
    glPopMatrix();

    /* Window refresh */
    glutSwapBuffers();
}



int main(int argc, char** argv)
{
    /* GLUT environment initialisation */
    glutInit(&argc, argv);
    glutInitDisplayMode(GLUT_DOUBLE | GLUT_RGB | GLUT_DEPTH);

    /* Window creation */
    glutInitWindowSize(800, 600);
    glutInitWindowPosition(240, 212);
    _windowNumber = glutCreateWindow(argv[0]);

    /* Keyboard interuptions */
    glutKeyboardFunc(key);
    glutSpecialFunc(special_keys);

    /* Callback function assignment */
    glutDisplayFunc(display);
    glutIdleFunc(no_events);
    glutReshapeFunc(reshape);

    /* Initialize Opengl states */
    glEnable (GL_DEPTH_TEST);
    glShadeModel(GL_SMOOTH);
    init_light();

    Vec4 eye = {0.f, 0.f, 4.f, 1.f};
    Vec4 up  = {0.f, 1.f, 0.f, 1.f};
    Vec4 aim = {0.f, 0.f, 0.f, 1.f};
    _cam = camera_init(eye, up, aim);

    //_mesh = mesh_init("bunny.off");
    //_mesh = mesh_init("buddha.off");
    _mesh = mesh_init("triceratops.off");

    /* Main event loop */
    glutMainLoop();

    return 0;
}
