#include "mesh.h"

#include "vector4.h"
#include "glassert.h"
#include <stdio.h>
#include <stdlib.h>
#include <math.h>

Mesh* mesh_init(char* filename)
{
    FILE* f = 0; // file declaration
    char c[3];   // 3 char table
    int dummy = 0;

    // open a read-only file
    f = fopen(filename, "r");
    // Check if its opened
    if(f == 0)
    {
        fprintf(stderr, "ERROR: can't open %s\n", filename);
        exit(-1);
    }

    // allocation for a Mesh structure
    Mesh* mesh = (Mesh*)malloc( sizeof(Mesh) );


    vec4_set(mesh->mat.ambient, 0.4f, 0.2f, 0.08f, 0.0f);
    vec4_set(mesh->mat.diffuse, 0.4f, 0.2f, 0.08f, 0.0f);
    vec4_set(mesh->mat.specular, 1.0f, 0.5f, 0.2f, 0.0f);
    mesh->mat.shininess = 90.f;

    // read the first 3 char of the file
    fread(c, sizeof(char), 3, f);
    // Check the file is an OFF file
    if(c[0] != 'O' || c[1] != 'F' || c[2] != 'F')
    {
        fprintf(stderr, "ERROR: wrong file format\n");
        exit(-1);
    }

    // read the number of vertices
    fscanf(f, "%d", &(mesh->nb_vertices));
    printf("nb vertices mesh: %d\n", mesh->nb_vertices);fflush(stdout);
    // Number of faces
    fscanf(f, "%d", &(mesh->nb_faces   ));
    printf("nb vertices mesh: %d\n", mesh->nb_faces);fflush(stdout);
    // number of edges
    fscanf(f, "%d", &dummy              );

    // memory allocation vertex coordinates
    mesh->verts = (float*)malloc( sizeof(float) * 3 * mesh->nb_vertices);

    // read the vertices x,y,z coordinates
    // compute center of gravity and max coordinate
    int i;
    float max_c = 0.f;
    Vec4 cog = {0.f, 0.f, 0.f, 0.f};
    for( i = 0; i < mesh->nb_vertices; i++)
    {
        Vec4 vert;

        fscanf(f, "%f", vert  );
        fscanf(f, "%f", vert+1);
        fscanf(f, "%f", vert+2);
        mesh->verts[i*3  ] = vert[0];
        mesh->verts[i*3+1] = vert[1];
        mesh->verts[i*3+2] = vert[2];
        max_c = fmaxf(fabsf(vert[0]), max_c);
        max_c = fmaxf(fabsf(vert[1]), max_c);
        max_c = fmaxf(fabsf(vert[2]), max_c);

        vec4_sum(vert, cog, cog);
    }

    vec4_mult_scalar(1.f/(float)mesh->nb_vertices, cog, cog);

    // Center and scale the mesh
    for( i = 0; i < mesh->nb_vertices; i++)
    {
        Vec4 vert = { mesh->verts[i*3], mesh->verts[i*3+1], mesh->verts[i*3+2] };

        vec4_diff(vert, cog, vert);
        vec4_mult_scalar(1.f/max_c, vert, vert);
        mesh->verts[i*3  ] = vert[0];
        mesh->verts[i*3+1] = vert[1];
        mesh->verts[i*3+2] = vert[2];
    }

    // memory allocation of the face index
    mesh->index = (int*)malloc( sizeof(int) * 3 * mesh->nb_faces);

    // read the face index
    for( i = 0; i < mesh->nb_faces; i++)
    {
        fscanf(f, "%d", &dummy                );
        fscanf(f, "%d", (mesh->index+i*3   )  );
        fscanf(f, "%d", (mesh->index+i*3 + 1) );
        fscanf(f, "%d", (mesh->index+i*3 + 2) );
    }

    // Allocate and compute the mesh normals
    mesh->normals = (float*)malloc( sizeof(float) * 3 * mesh->nb_vertices);
    for( i = 0; i < mesh->nb_vertices*3; i++)
        mesh->normals[i] = 0.f;

    int*   index = mesh->index;
    float* verts = mesh->verts;
    for( i = 0; i < mesh->nb_faces; i++)
    {
        int a = index[i*3  ];
        int b = index[i*3+1];
        int c = index[i*3+2];

        Vec4 v0 = {verts[a*3], verts[a*3+1], verts[a*3+2], 1.f};
        Vec4 v1 = {verts[b*3], verts[b*3+1], verts[b*3+2], 1.f};
        Vec4 v2 = {verts[c*3], verts[c*3+1], verts[c*3+2], 1.f};

        Vec4 e0;
        vec4_diff(v1, v0, e0);
        Vec4 e1;
        vec4_diff(v2, v0, e1);

        Vec4 n;
        vec4_cross_product(e0, e1, n);
        vec4_normalize(n);

        int j;
        for(j = 0; j < 3; j++)
        {
            mesh->normals[a*3+j] += n[j];
            mesh->normals[b*3+j] += n[j];
            mesh->normals[c*3+j] += n[j];
        }
    }

    // Normalize the sumed normals
    for(i = 0; i < mesh->nb_vertices; i++)
    {
        Vec4 n = {mesh->normals[i*3], mesh->normals[i*3+1], mesh->normals[i*3+2], 1.f};
        vec4_normalize(n);
        mesh->normals[i*3  ] = n[0];
        mesh->normals[i*3+1] = n[1];
        mesh->normals[i*3+2] = n[2];
    }
    // -------------------------------------

    // close the file
    fclose(f);

    return mesh;
}

void mesh_delete(Mesh* mesh)
{
    free( mesh->normals );
    free( mesh->verts   );
    free( mesh->index   );
    free( mesh          );
}

void mesh_draw_normals(Mesh* mesh, float size)
{
    GLboolean s;
    glGetBooleanv(GL_LIGHTING, &s);

    glDisable(GL_LIGHTING);
    glColor3f(0.2f, 0.1f, 0.8f);
    glBegin(GL_LINES);

    int i;
    for( i = 0; i < mesh->nb_vertices; i++)
    {
        Vec4 n  = {mesh->normals[i*3], mesh->normals[i*3+1], mesh->normals[i*3+2], 1.f};
        Vec4 v0 = {mesh->verts[i*3]  , mesh->verts  [i*3+1], mesh->verts  [i*3+2], 1.f};
        Vec4 v1;
        vec4_mult_scalar(size, n, n);
        vec4_sum(v0, n, v1);
        glVertex3f(v0[0], v0[1], v0[2]);
        glVertex3f(v1[0], v1[1], v1[2]);
    }
    glEnd();
    if(s) glEnable(GL_LIGHTING);
}

void mesh_draw(Mesh* mesh)
{
    materials_set_opengl_state(mesh->mat);
    glEnableClientState(GL_VERTEX_ARRAY);
    glVertexPointer(3, GL_FLOAT, 0, mesh->verts);

    glEnableClientState(GL_NORMAL_ARRAY);
    glNormalPointer(GL_FLOAT, 0, mesh->normals);

    glDrawElements(GL_TRIANGLES, mesh->nb_faces*3, GL_UNSIGNED_INT, mesh->index);

    glDisableClientState(GL_VERTEX_ARRAY);
    glDisableClientState(GL_NORMAL_ARRAY);

    GL_CHECK_ERRORS();
}
