#ifndef GL_ASSERT_H
#define GL_ASSERT_H

#include <GL/glu.h>
#include <stdio.h>
#include <assert.h>
#include <stdbool.h>

/** @brief debugging tool for opengl calls

    This file provide macros in order to check easily openGl API calls.
    use it on every call to ensure better correctness of your programm :
    @code
    #include "glassert.h"

    glAssert( glmyAPICall() )
    @endcode

    @warning never use a glAssert() on a begin() directive.
    glGetError is not allowed inside a glBegin gEnd block !

    One can track down opengl errors with GL_CHECK_ERRORS(); which is usefull if
    the project don't use glAssert() everywhere. In this case a glAssert() can
    be triggered by previous error and won't give valuable insight on where the
    problem is located and from what opengl's primitive call.
*/

#define __TO_STR(x) __EVAL_STR(x)
#define __EVAL_STR(x) #x

#define glAssert(code) do{code; int l = __LINE__;\
   int err = (int)glGetError(); \
                if (err != GL_NO_ERROR)\
                { \
                  fprintf(stderr, "OpenGL error : %s\n", __FILE__);\
                  fprintf(stderr, "ligne : %d \n", l);\
                  fprintf(stderr, "Code source : %s \n", __TO_STR(code));\
                  fprintf(stderr, "Glut error : %s ( %d ) \n", (char*)gluErrorString(err), err);\
                  assert(false); \
              }\
}while(false)

// -----------------------------------------------------------------------------

#define GL_CHECK_ERRORS() \
    do{ int l = __LINE__;\
       int err = (int)glGetError(); \
                    if (err != GL_NO_ERROR)\
                    { \
                      fprintf(stderr, "OpenGL error : %s\n", __FILE__);\
                      fprintf(stderr, "ligne : %d \n", l);\
                      fprintf(stderr, "Glut error : %s ( %d ) \n", (char*)gluErrorString(err), err);\
                      assert(false); \
                  }\
    }while(false)
#endif
