//      ReinhardLocalOperator.cpp
//      
//      Copyright 2010 Jérémy Laumon <jeremy.laumon@gmail.com>
//      
//      This program is free software; you can redistribute it and/or modify
//      it under the terms of the GNU General Public License as published by
//      the Free Software Foundation; either version 2 of the License, or
//      (at your option) any later version.
//      
//      This program is distributed in the hope that it will be useful,
//      but WITHOUT ANY WARRANTY; without even the implied warranty of
//      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//      GNU General Public License for more details.
//      
//      You should have received a copy of the GNU General Public License
//      along with this program; if not, write to the Free Software
//      Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
//      MA 02110-1301, USA.


#include <ReinhardLocalOperator.h>
#include <Exceptions.h>
#include <math.h>

#define OPERATOR_NAME "Reinhard Local Operator"

Q_EXPORT_PLUGIN2(TT_ReinhardLocalOperator, ReinhardLocalOperatorFactory)

/**
 * \class ReinhardLocalOperator
 * \brief The ReinhardLocalOperator class is an implementation of Erik 
 * Reinhard's local tone mapping operator.
 * 
 * This class implements ToneMappingOperator interface. 
 * 
 * FFTW library is used to calculate convolutions efficiently.
 */

ReinhardLocalOperator::ReinhardLocalOperator() :
    inputImage(NULL),
    outputImage(NULL),
    width(0),
    height(0),
    keyValue(0.18),
    sharpening(8.0)
{
    for(int k=0; k<9; ++k)
    {
        responses[k] = NULL;
    }
}

ReinhardLocalOperator::~ReinhardLocalOperator()
{
    for(int k=0; k<9; ++k)
    {
        delete[] responses[k];
    }
}

/**
 * Returns the operator name: Reinhard Local Operator.
 */
QString ReinhardLocalOperator::name() const
{
    return tr(OPERATOR_NAME);
}

void ReinhardLocalOperator::setupUi(QWidget* parent)
{
    ui.setupUi(parent);
    
    connect(ui.keyValueSlider, SIGNAL(sliderReleased()), this, SLOT(toneMap()));
    connect(ui.keyValueSlider, SIGNAL(valueChanged(int)), this, SLOT(updateKeyValue(int)));
    
    connect(ui.sharpeningSlider, SIGNAL(sliderReleased()), this, SLOT(toneMap()));
    connect(ui.sharpeningSlider, SIGNAL(valueChanged(int)), this, SLOT(updateSharpening(int)));
}

const HdrImage* ReinhardLocalOperator::getToneMappedImage() const
{
    return outputImage;
}

/**
 * Sets the image to be tone mapped.
 * 
 * This function also initializes the operator by computing :
 * -# the average luminance
 * -# the 9 required gaussian filters
 * -# the 9 convolutions of the image by the gaussian filters (called responses)
 * 
 * and then calls toneMap().
 * 
 * For improved efficiency, the convolutions are computed in the Fourier domain. 
 * 
 */
void ReinhardLocalOperator::setImage(const HdrImage* _inputImage)
{
    if (!_inputImage->hasY())
        throw Exception(tr("Image passed to %1 does not contains Y data. Cannot turn water into wine.").arg(name()));
    
    QTime t;
    t.start();
    
    inputImage = _inputImage;
    QSize size = inputImage->size();
    width = size.width();
    height = size.height();
    int Y = inputImage->YIndex();
    int length =  width*height;
    double delta = 0.0001;
    fftw_plan plan;
    fftw_complex* image = NULL;
    fftw_complex* filters[9] = {NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL};
    
    delete outputImage;
    outputImage = new HdrImage(*inputImage);

	// delete data from potential previous image
    for(int k=0; k<9; ++k)
    {
        delete[] responses[k];
    }
    
    
    // calculate average luminance
    //TO COMPLETE
	
    
    // allocate memory for image fft and filters ffts
    image = new fftw_complex[length];
    for(int k=0; k<9; ++k)
    {
        filters[k] = new fftw_complex[length];
        for(int i=0; i<length; ++i)
        {
            filters[k][i][0] = 0.0; // re
            filters[k][i][1] = 0.0; // im
        }
    }

    // generate filters data
	// TO COMPLETE
    
    // compute filters ffts
	// TO COMPLETE, see fftw_plan_dft_2d() and fftw_execute()
    
    // copy image data
    for(int i=0; i<height; ++i)
    {
        for(int j=0; j<width; ++j)
        {
            image[i*width + j][0] = (*inputImage)[i][j][Y];
            image[i*width + j][1] = 0.0;
        }
    }
        
    // compute image fft
	// TO COMPLETE, see fftw_plan_dft_2d() and fftw_execute()
    
    // convolve image and filters
	// TO COMPLETE
    
    // compute responses inverse ffts
	// TO COMPLETE, see fftw_plan_dft_2d() and fftw_execute()
        
    fftw_destroy_plan(plan);
    delete[] image;
    for(int k=0; k<9; ++k)
        delete[] filters[k];
    
    ui.keyValueSlider->setValue(18); // = 0.18
    ui.sharpeningSlider->setValue(8); 
    
    msg = tr("Operator Init: %1 ms").arg(t.elapsed());
    
    toneMap();
}

/**
 * Computes the convolution of \a f1 by \a f2 in the Fourier domain.
 * A convolution in the Fourier domain is actually a complex multiplication.
 */
fftw_complex* ReinhardLocalOperator::convolveFft(fftw_complex* f1, fftw_complex* f2)
{
    int length = width*height;
    fftw_complex* res = reinterpret_cast<fftw_complex*>(fftw_malloc(sizeof(fftw_complex)*length));

	//TO COMPLETE
	
    return res;
}

/**
 * Tone maps the image.
 * 
 * This function emit imageUpdated() when tone mapping is finished.
 */
void ReinhardLocalOperator::toneMap()
{
    if(inputImage)
    {
        QTime t;
        t.start();
        
        int Y = inputImage->YIndex();
        const float threshold = 0.05;

		//for each pixel, find the appropriate filter to modulate the luminance
		//in the out image
		//TO COMPLETE
               
        
        emit message(msg + tr("  Tone Mapping: %1 ms").arg(t.elapsed()));
                
        emit imageUpdated();
    }
}

/**
 * Reads the key value from the user interface.
 * 
 * This function is typically called when the key value slider is moved.
 */
void ReinhardLocalOperator::updateKeyValue(int value)
{
    keyValue = double(value)/100.0;
    ui.keyValue->setText(QString("%1").arg(keyValue, 0, 'f', 2));
}

/**
 * Reads the sharpening parameter from the user interface.
 * 
 * This function is typically called when the sharpening parameter slider is moved.
 */
void ReinhardLocalOperator::updateSharpening(int value)
{
    sharpening = double(value);
    ui.sharpening->setText(QString("%1").arg(sharpening));
}


//----------------------------------------------------------------------

/**
 * \class ReinhardLocalOperatorFactory
 * \brief The ReinhardLocalOperatorFactory class creates instances of ReinhardLocalOperator.
 * 
 * This class is an implementation of ToneMappingOperatorFactory.
 */

/**
 * Returns a newly created ReinhardLocalOperator.
 */
ToneMappingOperatorPtr ReinhardLocalOperatorFactory::createOperator() const
{
    return ToneMappingOperatorPtr(new ReinhardLocalOperator);
}

/**
 * Returns the operator name: Reinhard Local Operator.
 */
QString ReinhardLocalOperatorFactory::operatorName() const
{
    return tr(OPERATOR_NAME);
}
